#ifndef _UI_H_
#define  _UI_H_
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <sys/stat.h>
#include <math.h>
#include <pthread.h>
#include "ctx.h"


/////////////////////////////////////////////
typedef struct _Ui       Ui;
typedef void (*ui_fun)(Ui *ui);
typedef void (*ui_data_finalize)(void *data);

void ui_add_key_binding   (Ui *ui, const char *key, const char *action, const char *label);


void *ui_last_id (Ui *ui);
void ui_focus (Ui *ui, void *id);
void *ui_focused_id (Ui *ui);

// create a Ui instance from ctx, should be paired
// with ui_destroy
Ui *ui_new                (Ctx *ctx);
void ui_destroy           (Ui *ui);

// like ui_new, can take NULL for ctx - should not
// be ui_destroy'ed - reuses the system pre-existing
// Ui instance.
Ui *ui_host               (Ctx *ctx);

Ui *ui_new2 (Ctx *ctx);

char *s0il_load_file      (Ui *ui, const char *path, int *ret_length);

// do an action, or go to a specified view - if activating a new
// view the current view is first pushed on the view stack,
//   built-in/reserved commands: back kb-hide kb-show kb-collapse exit
//
// names starting with / are taken to be local file system paths
// based on mime-type a registered view is used for rendering.

void ui_do              (Ui *ui, const char *name);

void ui_start_frame       (Ui *ui);

// should be called after all ui functions in a frame are done with
// multiple different sections panels might exist through ui_set_dim
// which changes the flow widges are layouted in
void ui_end_frame         (Ui *ui);


void ui_focus_next        (Ui *ui);
void ui_focus_prev        (Ui *ui);

typedef enum
{
  UI_DEFAULT    = 0,
  UI_INLINE     = (1<<0), // as opposed to expand + newline
  UI_REVERSE    = (1<<1), // reverses foregrond and background
  UI_NO_BASE    = (1<<2),
  UI_NO_LABEL   = (1<<3),
  UI_SQUARE     = (1<<4), // as opposed to rounded corners
  UI_LEFT_ALIGN = (1<<5),
  UI_ICON       = (1<<6), // expand label to be an icon - 
                          // do this by a "icon: prefix instead?
  UI_VERTICAL   = (1<<7),
  UI_NO_ALPHA   = (1<<8),


} UiFlag;

void ui_flags (Ui *ui, UiFlag flags);

int  ui_button (Ui *ui, const char *label);
bool ui_toggle (Ui *ui, const char *label, bool value);
void ui_title  (Ui *ui, const char *string);
void ui_text (Ui *ui, const char *string);
void ui_textf (Ui *ui, const char *format, ...);


void ui_choice_add (Ui *ui, const char *label, int value);
int ui_choice (Ui *ui, const char *label, int value);

// todo ui_radio int(ui,label,int set)
// todo ui_expander
// void ui_control_no(ui)

void ui_seperator(Ui *ui);
void ui_newline(Ui *ui);

typedef enum
{
  UI_COLOR_FULL,
  UI_COLOR_BASE,
  UI_COLOR_INK
} UiColorType;

void
ui_color_compute (Ui *ui, UiColorType type,
                  float value, float temperature, float alpha, float *rgba);
void
ui_color (Ui *ui, UiColorType type,
          float value, float temperature, float alpha);
void
ui_color_ink (Ui *ui, float value, float temperature);
void
ui_color_base (Ui *ui, float value, float temperature);


char *ui_entry(Ui *ui, const char *label, const char *fallback, const char *value);
int ui_entry_realloc(Ui *ui, const char *label, const char *fallback, char **strptr);

float ui_slider(Ui *ui, const char *label, float min, float max, float step, float value);
void ui_slider_float (Ui *ui, const char *label, float *val, float min, float max, float step);
void ui_slider_int (Ui *ui, const char *label, int *val, int min, int max, int step);
void ui_slider_uint8 (Ui *ui, const char *label, uint8_t *val, uint8_t min, uint8_t max, uint8_t step);
void ui_slider_uint16 (Ui *ui, const char *label, uint16_t *val, uint16_t min, uint16_t max, uint16_t step);
void ui_slider_uint32 (Ui *ui, const char *label, uint32_t *val, uint32_t min, uint32_t max, uint32_t step);
void ui_slider_int8 (Ui *ui, const char *label, int8_t *val, int8_t min, int8_t max, int8_t step);
void ui_slider_int16 (Ui *ui, const char *label, int16_t *val, int16_t min, int16_t max, int16_t step);
void ui_slider_int32 (Ui *ui, const char *label, int32_t *val, int32_t min, int32_t max, int32_t step);


float ui_slider_coords(Ui *ui, void *id,
                       float x, float y, float width, float height,
                       float min_val, float max_val, float step, float value);
int ui_button_coords(Ui *ui, float x, float y, float width, float height,
                     const char *label, void *id);
char * ui_entry_coords(Ui *ui, void *id, float x, float y, float w, float h,
                       const char *fallback, const char *value);
int ui_choice_coords (Ui *ui, float x, float y, float width, float height,
                      const char *label, void *id, int value);



float ui_get_font_size    (Ui *ui);
void  ui_cb_do            (CtxEvent *event, void *data1, void *data2);
Ctx  *ui_ctx              (Ui *ui);
float ui_x                (Ui *ui);
float ui_y                (Ui *ui);
void  ui_move_to          (Ui *ui, float x, float y);

void ui_set_dim(Ui *ui, float x0, float y0, float width, float height);

void ui_color_set_warm  (Ui *ui, const float *rgb);
void ui_color_get_warm  (Ui *ui, float *rgb);
void ui_color_set_cool  (Ui *ui, const float *rgb);
void ui_color_get_cool  (Ui *ui, float *rgb);
void ui_color_set_light (Ui *ui, const float *rgb);
void ui_color_get_light (Ui *ui, float *rgb);
void ui_color_set_dark  (Ui *ui, const float *rgb);
void ui_color_get_dark  (Ui *ui, float *rgb);


typedef enum {
  UI_SELECTED_BG,
  UI_SELECTED_BORDER,
  UI_TEXT,
  UI_ACTIVE,
  UI_INTERACTIVE,
} UiStyle;

int ui_main (Ui *ui_in, void (*fun)(Ui *ui_in, float delta_s, void *data), void *data);

#endif

