#include <unistd.h>
#include <stdarg.h>
#include "ui.h"


#define UI_MAX_WIDGETS  128
#define UI_MAX_OVERLAYS 4
#define UI_MAX_HISTORY  16

#define LERP(a,b,dt) ((a)*(1.0-dt)+(dt)*(b))

float col_white[3]={1.0,1.0,1.0};
float col_black[3]={0.0,0.0,0.0};
float col_warm[3]={1.0,0.6,0.0};
float col_cool[3]={0.0,0.5,1.0};


#define STYLE_HOVERED \
          do {ui_color_ink (ui, 0.7, -0.4f);} while (0)
#define STYLE_FOCUSED \
          do {ui_color_ink (ui, 0.6, -0.7f);} while (0)
#define STYLE_TEXT \
          do {ui_color_ink (ui, 0.7, 0.0f);} while (0)
#define STYLE_TEXT_INTERACTIVE \
          do {ui_color_ink (ui, 0.7, -0.2f);} while (0)
#define STYLE_TEXT_ACTIVE \
          do {ui_color_ink (ui, 0.7, -0.6f);} while (0)


#define STYLE_WIDGET_BORDER \
          do {ui_color_ink (ui, 0.5, 0.0f);} while (0)
#define STYLE_WIDGET_BORDER_HOVERED \
          do {ui_color_ink (ui, 0.25, -0.1f);} while (0)
#define STYLE_WIDGET_BORDER_FOCUSED \
          do {ui_color_ink (ui, 0.6, -0.7f);} while (0)

#define STYLE_WIDGET_BASE \
          do {ui_color_base (ui, 0.6, 0.2f);} while (0)

#define STYLE_WIDGET_BASE_FOCUSED \
          do {ui_color_base (ui, 0.5, -0.4f);} while (0)
#define STYLE_WIDGET_BASE_HOVERED  \
          do {ui_color_base (ui, 0.7, -0.3f);} while (0)


void ui_hue_to_rgb (float hue, float *rgb)
{
  float rainbow[6][3]={
    {1.0,0.0,0.0},
    {1.0,1.0,0.0},
    {0.0,1.0,0.0},
    {0.0,1.0,1.0},
    {0.0,0.0,1.0},
    {1.0,0.0,1.0},
  };
 
  while (hue < 0)
    hue += 1.0f;
  while (hue > 1.0)
    hue -= 1.0f;

  int ia = hue * 5.0f;

  if (ia > 4) ia =4;
  float dt = hue * 5 - ia;
  for (int c = 0; c < 3; c++)
    rgb[c] = rainbow[ia][c] * (1.0f-dt) + rainbow[ia+1][c] * (dt);
}

// value range 0.0-1.0
// temperature range -1 to 1
// to achieve pure white or black temperature must be 0
//
static inline void
vta_to_rgba (float value, float temperature, float alpha, float *rgba)
{
  if (value > 1.0f)
    value = 1.0f;
  if (value < 0.0f)
    value = 0.0f;
  if (temperature > 1.0f)
    temperature = 1.0f;
  if (temperature < -1.0f)
    value = -1.0f;

  for (int i = 0; i < 3; i++)
    rgba[i] = LERP(col_black[i], col_white[i], value);

  if (temperature > 0)
  {
    if (temperature>1.0) temperature = 1.0f;
    for (int i = 0; i < 3; i++)
      rgba[i] = LERP(rgba[i], col_warm[i], temperature);
  }
  else if (temperature < 0)
  {
    temperature = -temperature;
    if (temperature>1.0) temperature = 1.0f;
    for (int i = 0; i < 3; i++)
      rgba[i] = LERP(rgba[i], col_cool[i], temperature);
  }

  for (int i = 0; i < 3; i++)
    rgba[i] = powf (rgba[i], 2.4f);
  rgba[3] = alpha;
}

void
ui_color_set_warm (Ui *ui, const float *rgb)
{
  for (int i = 0; i < 3; i++)
    col_warm[i] = powf (rgb[i],1.0/2.2f);
}

void
ui_color_get_warm (Ui *ui, float *rgb)
{
  for (int i = 0; i < 3; i++)
    rgb[i] = col_warm[i];
}


void
ui_color_set_cool (Ui *ui, const float *rgb)
{
  for (int i = 0; i < 3; i++)
    col_cool[i] = powf (rgb[i],1.0/2.2f);
}

void
ui_color_get_cool (Ui *ui, float *rgb)
{
  for (int i = 0; i < 3; i++)
    rgb[i] = col_cool[i];
}

void
ui_color_set_darkest (Ui *ui, const float *rgb)
{
  for (int i = 0; i < 3; i++)
    col_black[i] = powf (rgb[i],1.0/2.2f);
}

void
ui_color_get_darkest (Ui *ui, float *rgb)
{
  for (int i = 0; i < 3; i++)
    rgb[i] = col_black[i];
}

void
ui_color_set_brightest (Ui *ui, const float *rgb)
{
  for (int i = 0; i < 3; i++)
    col_white[i] = powf (rgb[i],1.0/2.2f);
}

void
ui_color_get_brightest (Ui *ui, float *rgb)
{
  for (int i = 0; i < 3; i++)
    rgb[i] = col_white[i];
}

static float brightness = 0.23f;
void ui_set_brightness (Ui *ui, float l)
{
   brightness = l/100.0f;
}
float ui_get_brightness (Ui *ui)
{
   return brightness * 100.0f;
}

static float contrast = 1.0f;
void ui_set_contrast (Ui *ui, float l)
{
   contrast = l;
}
float ui_get_contrast (Ui *ui)
{
   return contrast;
}

static float saturation = 1.3f;
void ui_set_saturation (Ui *ui, float l)
{
   saturation = l/100.0f;
}
float ui_get_saturation (Ui *ui)
{
   return saturation * 100.0f;
}

static float base_saturation = 1.0f;
void ui_set_base_saturation (Ui *ui, float l)
{
   base_saturation = l/100.0f;
}
float ui_get_base_saturation (Ui *ui)
{
   return base_saturation * 100.0f;
}

void
ui_color_compute (Ui *ui, UiColorType type,
                  float value, float temperature, float alpha, float *rgba)
{
  switch (type)
  {
    case UI_COLOR_FULL:
      vta_to_rgba (value, temperature, alpha, rgba);
      break;
    case UI_COLOR_BASE:
      {
      float v = brightness - 0.25 + value/2;
      float t = 0.0f;

      //v = (v-(brightness)) * contrast + (brightness);
      t = base_saturation * saturation * temperature;

      vta_to_rgba (v, t, alpha, rgba);
      }
      break;
    case UI_COLOR_INK:
      {
  float t = 0.0f;
  float v;

  value = (value - 0.05) * contrast + 0.05;

  t = saturation * temperature;

  if (brightness > 0.7)
  {
    v = (1.0 - value) * brightness;
  }
  else
  {
    v = brightness + (1.0-brightness) * (value);
  }

      vta_to_rgba (v, t, alpha, rgba);
      }
      break;
  }
}
               
void
ui_color (Ui *ui, UiColorType type,
          float value, float temperature, float alpha)
{
  float rgba[4];
  ui_color_compute (ui, type, value, temperature, alpha, rgba);
  ctx_rgba (ui_ctx (ui), rgba[0], rgba[1], rgba[2], rgba[3]);
}

void ui_color_ink_alpha (Ui *ui, float value, float temperature, float alpha)
{
  ui_color (ui, UI_COLOR_INK, value, temperature, alpha);
}

void ui_color_ink (Ui *ui, float value, float temperature)
{
  ui_color_ink_alpha (ui, value, temperature, 1.0f);
}

void ui_color_base_alpha (Ui *ui, float value, float temperature, float alpha)
{
  ui_color (ui, UI_COLOR_BASE, value, temperature, alpha);
}
void ui_color_base (Ui *ui, float value, float temperature)
{
  ui_color_base_alpha (ui, value, temperature, 1.0f);
}

typedef struct _UiWidget UiWidget;


typedef enum {
  ui_type_none = 0,
  ui_type_button,
  ui_type_slider,
  ui_type_entry,
  ui_type_choice,
  ui_type_colorpopup,
} ui_type;

typedef enum {
  ui_state_default = 0,
  ui_state_hot,
  ui_state_lost_focus,
  ui_state_commit,
} ui_state;

struct _UiWidget {
  void *id; // unique identifier
  ui_type type;
  UiFlag flags;
  float x;    // bounding box of widget
  float y;    // (ink rect might be beyond, but this is the event zone)
  float width;
  float height;

  uint8_t state;           // < maybe refactor to be named bools instead?
  uint8_t overlay_no;
  bool visible : 1;        // set on first registration/creation/re-registration
  bool fresh : 1;          // set on first registration/creation/re-registration
                           // of widget - to initialize stable values
  bool  hovered : 1;
  int active ; // cheating and used as a counter for stages in choice

  float min_val;           //  used by slider
  float max_val;           //
  float step;              //
  float float_data;        //
  float float_data_target; //
};


typedef struct _UiOverlay UiOverlay;

struct _UiOverlay
{
  Ctx *drawlist;
  float x0, y0, x, y, width, height; // these are the values to restore
                                     //
  float a, b, c, d, e, f, g, h, i;
  void *focused_id;
};

struct _Ui {
  Ctx *ctx;
  Ctx *ectx;

  ui_fun fun;
  int delta_ms;
  void *focused_id;
  void *last_id;
  float font_size_px;
  float view_elapsed;
  int frame_no;
  bool draw_tips;

  void *edit_slider;

  UiOverlay overlay[UI_MAX_OVERLAYS];
  int overlay_no;
  int max_overlay;
  int prev_max_overlay;

  float osk_focus_target;
  bool owns_ctx;
  void *active_id;
  int activate;
  float x0;
  float y0;
  float width;
  float height;
  float line_height;
  float x;
  float y;
  int cursor_pos;
  int selection_length;
  int drawn_overlays;

#define UI_MAX_CHOICES 256

  int widget_count;
  UiWidget widgets[UI_MAX_WIDGETS];
  char temp_text[256];
  char backup_text[256];

  bool focus_first;
  int queued_next;

  float pointer_x_global;
  float pointer_y_global;

  float pointer_x;
  float pointer_y;

  char *choice_name;
  float choice_x;
  char *choice_names[UI_MAX_CHOICES];
  int choice_values[UI_MAX_CHOICES];
  int n_choices;

  UiFlag flags;
};

int ctx_osk_mode = 0;

///////////////////////////////////////////////////////////////////////

#define UI_ID_STR(label) ((void *)(size_t)(ctx_strhash(label) | 1))
#define UI_ID ((void *)(__LINE__ * 2))
#define em (ui->font_size_px)
int demo_mode = 1;
void ui_keyboard(Ui *ui);
static bool osk_captouch = false;

////////////////////////////////////////////////////////
char *s0il_load_file(Ui *ui, const char *path, int *ret_length) {
  FILE *file = fopen(path, "rb");
  char *data = NULL;
  if (file) {
    fseek(file, 0, SEEK_END);
    long length = ftell(file);
    fseek(file, 0, SEEK_SET);
    if (ret_length)
      *ret_length = length;
    data = malloc(length + 1);
    if (data) {
      if (!fread(data, length, 1, file))
      {
        fclose(file);
        free(data);
        return NULL; // XXX : hapazard?
      }
      fclose(file);
      ((char *)data)[length] = 0;
    }
  } else {
    printf("load failing for %s\n", path);
  }
  return data;
}

static UiWidget *ui_widget_by_id(Ui *ui, void *id);

void ui_increase_or_focus_next(Ui *ui);
void ui_decrease_or_focus_prev(Ui *ui);
void ui_increase_or_focus_right(Ui *ui);
void ui_decrease_or_focus_left(Ui *ui);
void ui_focus_direction (Ui *ui, int direction);

int widget_activate (Ctx *ctx, void *a)
{
  Ui *ui = a;
  UiWidget *widget = ui_widget_by_id(ui, ui->focused_id);
  widget->active = 1;
  //ui->focused_id = widget->id;
  ctx_queue_draw (ctx);
  return 0;
}

void
ui_do(Ui *ui, const char *action) {
  ctx_queue_draw(ui->ctx);
  // printf ("ui_do: %s\n", action);
  if (!strcmp(action, "close-overlay")) {
    UiWidget *widget = ui_widget_by_id(ui, ui->overlay[ui->max_overlay-1].focused_id);
    if (widget && widget->active) 
    {
      widget->active = 0;
      ctx_queue_draw (ui->ctx);
    }
  }
  else if (!strcmp(action, "exit")) {
    ctx_exit(ui->ctx);
    // we proxy some keys, since this makes binding code simpler
  } else if (!strcmp(action, "backspace") || !strcmp(action, "return") ||
             !strcmp(action, "left") || !strcmp(action, "escape") ||
             !strcmp(action, "right") || !strcmp(action, "space")) {
    ctx_key_press(ui->ctx, 0, action, 0);
  } else if (!strcmp(action, "activate-slider")) { // same as activate without slider edit bits
    UiWidget *widget = ui_widget_by_id(ui, ui->focused_id);
    if (ui->focused_id)
      ui->activate = 1;
    if (widget && (widget->type == ui_type_entry))
      ui_do(ui, "kb-show");
    ui->active_id = NULL;
  } else if (!strcmp(action, "activate")) {
    UiWidget *widget = ui_widget_by_id(ui, ui->focused_id);
    if (ui->focused_id)
      ui->activate = 1;
    if (widget && (widget->type == ui_type_slider))
    {
      ui->edit_slider = ui->focused_id;
    }
    if (widget && (widget->type == ui_type_choice ||
                   widget->type == ui_type_colorpopup))
    {
      ui->activate = 0;
      ctx_add_idle (ui->ectx, widget_activate, ui);
    }
    if (widget && (widget->type == ui_type_entry))
      ui_do(ui, "kb-show");
    ui->active_id = NULL;
  } else if (!strcmp(action, "increase-or-focus-next")) {
    ui_increase_or_focus_next(ui);
  } else if (!strcmp(action, "decrease-or-focus-previous")) {
    ui_decrease_or_focus_prev(ui);
  } else if (!strcmp(action, "increase-or-focus-right")) {
    ui_increase_or_focus_right(ui);
  } else if (!strcmp(action, "decrease-or-focus-left")) {
    ui_decrease_or_focus_left(ui);
  } else if (!strcmp(action, "focus-next")) {
    ui_focus_next(ui);
  } else if (!strcmp(action, "focus-previous")) {
    ui_focus_prev(ui);
  } else if (!strcmp(action, "focus-right")) {
    ui_focus_direction (ui, 0);
  } else if (!strcmp(action, "focus-down")) {
    ui_focus_direction (ui, 1);
  } else if (!strcmp(action, "focus-left")) {
    ui_focus_direction (ui, 2);
  } else if (!strcmp(action, "focus-up")) {
    ui_focus_direction (ui, 3);
  } else if (!strcmp(action, "kb-collapse")) {
    ctx_osk_mode = 1;
  } else if (!strcmp(action, "kb-show")) {
    ctx_osk_mode = 2;
  } else if (!strcmp(action, "kb-hide")) {
    ctx_osk_mode = 0;
  } else {
  }
}

void ui_cb_do(CtxEvent *event, void *a, void *b)
{
  Ui *ui = a;
  const char *target = b;
  event->stop_propagate = 1;
  ui_do(ui, target);
}

static Ui *def_ui = NULL;

Ui *ui_new(Ctx *ctx) {
  Ui *ui = calloc(1, sizeof(Ui));
  if (!def_ui) {
    def_ui = ui;
  }
  ui->ctx = ui->ectx = ctx;

  ui->focus_first = true;

  ui->osk_focus_target = 0.18f;
  osk_captouch = true;

  float width = ctx_width(ctx);
  float height = ctx_height(ctx);

  ui->width = width;
  ui->height = height;

  ui->font_size_px = ctx_get_font_size (ctx);

  return ui;
}

Ui *ui_for_ctx (Ctx *ctx)
{
   Ui *ret = NULL;
   const char *old = ctx_get_string (ctx, SQZ_ui);
   if (old)
   {
     void *ptr;
     memcpy (&ptr, old, sizeof (ptr));
     return ptr;
   }
   ret = ui_new (ctx);
   ctx_set_blob (ctx, SQZ_ui, (char*)&ret, sizeof (void*));
   return ret;
}

static UiWidget *ui_widget_by_id(Ui *ui, void *id) {
  for (int i = 0; i < ui->widget_count; i++) {
    if (ui->widgets[i].id == id)
      return &ui->widgets[i];
  }
  return NULL;
}
static void ui_set_focus(Ui *ui, UiWidget *widget) {
  ctx_queue_draw(ui->ctx);
  for (int i = 0; i < ui->widget_count; i++)
    if (ui->focused_id) {
      UiWidget *old_widget = ui_widget_by_id(ui, ui->focused_id);
      if (old_widget == widget) {
        ctx_queue_draw(ui->ctx);
        return;
      }
      if (old_widget) {
        old_widget->state = ui_state_lost_focus;
      }
      if (ui->active_id) {
        old_widget->state = ui_state_commit;
        ui->active_id = NULL;
        ui_do(ui, "kb-hide");
      }
      ui->active_id = NULL;
      ui->focused_id = NULL;
    }
  if (widget) {
    ui->focused_id = widget->id;
    widget->state = ui_state_hot;
  }
  ctx_queue_draw(ui->ctx);
}

void ui_focus_next(Ui *ui)
{
  bool found = false;
  ctx_queue_draw(ui->ctx);

  for (int i = 0; i < ui->widget_count; i++) {
    if ((found || !ui->focused_id) && ui->widgets[i].overlay_no == ui->max_overlay)
    {
      ui->focused_id = ui->widgets[i].id;
      ctx_queue_draw(ui->ctx);
      return;
    }
    if (ui->widgets[i].id == ui->focused_id) {
      found = true;
    }
  }
  ui->focused_id = NULL;
}

void ui_focus_prev(Ui *ui)
{
  bool found = false;
  ctx_queue_draw(ui->ctx);
  for (int i = ui->widget_count - 1; i >= 0; i--) {
    if ((found || !ui->focused_id) && ui->widgets[i].overlay_no == ui->max_overlay) {
      ui->focused_id = ui->widgets[i].id;
      ctx_queue_draw(ui->ctx);
      return;
    }
    if (ui->widgets[i].id == ui->focused_id) {
      found = true;
    }
  }
  ui->focused_id = NULL;
}

void ui_focus_direction (Ui *ui, int direction)
{

  UiWidget *widget = NULL;
  ctx_queue_draw(ui->ctx);

  for (int i = ui->widget_count - 1; i >= 0; i--) {
    if (ui->widgets[i].id == ui->focused_id) {
      widget = &ui->widgets[i];
    }
  }

  if (!widget)
    return;

  float x = widget->x + widget->width / 2;
  float y = widget->y + widget->height / 2;

  float dx = 0.0;
  float dy = 0.0;

  float sx = 1;
  float sy = 1;

  switch (direction)
  {
    case 0: dx = 2.0f; sx = 150; break;
    case 1: dy = 2.0f; sy = 150; break;
    case 2: dx = -2.0f; sx = 150; break;
    case 3: dy = -2.0f; sy = 150; break;
  }


  float dist_origin_sq = 0.0f;
  int max_steps = 1000;
  for (int s = 0; s < max_steps; s++)
  {
    dist_origin_sq += sx*dx*dx + sy*dy*dy;
    x += dx;
    y += dy;

    float best_dist = dist_origin_sq;
    UiWidget *best_w = NULL;

    for (int i = ui->widget_count - 1; i >= 0; i--) {
      {
        UiWidget *w = &ui->widgets[i];
        if (w->overlay_no == ui->prev_max_overlay)
        {
#define PW2(a) ((a)*(a))
        for (float u = 0.0f; u < 1.0; u+= 0.03)
        for (float v = 0.0f; v < 1.0; v+= 0.03)
        {
        float dist = sx*PW2((w->x + w->width * u)-x) +
                     sy*PW2((w->y + w->height * v)-y);


        if (dist < best_dist)
        {
          best_dist = dist;
          best_w = w;
        }

        }
#undef PW2
      }
      }
  }
    if (best_w && best_w != widget)
    {
      ui->focused_id = best_w->id;
      ctx_queue_draw(ui->ctx);
      return;
    }
  }



}

void ui_increase_or_focus_right(Ui *ui)
{
  UiWidget *widget = NULL;
  for (int i = ui->widget_count - 1; !widget && i >= 0; i--) {
    if (ui->widgets[i].id == ui->focused_id) {
      widget = &ui->widgets[i];
    }
  }
  if (!widget)
    return;
  if (widget->type == ui_type_slider)
  {
    ui_do (ui, "activate-slider");
  }
  else
  ui_focus_direction(ui,0);
}

void ui_decrease_or_focus_left(Ui *ui)
{
  UiWidget *widget = NULL;
  for (int i = ui->widget_count - 1; !widget && i >= 0; i--) {
    if (ui->widgets[i].id == ui->focused_id) {
      widget = &ui->widgets[i];
    }
  }
  if (!widget)
    return;
  if (widget->type == ui_type_slider)
  {
    ui_do (ui, "activate-slider");
  }
  else
  ui_focus_direction(ui, 2);
}

void ui_increase_or_focus_next(Ui *ui)
{
  UiWidget *widget = NULL;
  for (int i = ui->widget_count - 1; !widget && i >= 0; i--) {
    if (ui->widgets[i].id == ui->focused_id) {
      widget = &ui->widgets[i];
    }
  }
  if (!widget)
    return;
  if (widget->type == ui_type_slider)
  {
    ui_do (ui, "activate-slider");
  }
  else
  ui_focus_next(ui);
}

void ui_decrease_or_focus_prev(Ui *ui)
{
  UiWidget *widget = NULL;
  for (int i = ui->widget_count - 1; !widget && i >= 0; i--) {
    if (ui->widgets[i].id == ui->focused_id) {
      widget = &ui->widgets[i];
    }
  }
  if (!widget)
    return;
  if (widget->type == ui_type_slider)
  {
    ui_do (ui, "activate-slider");
  }
  else
  ui_focus_prev(ui);
}

int widget_unactivate (Ctx *ctx, void *data)
{
  UiWidget *widget = data;
  widget->active = 0;
  return 0;
}

static void ui_slider_key_press(CtxEvent *event, void *userdata,
                                void *userdata2) {

  Ui *ui = userdata;
  UiWidget *widget = userdata2;

  const char *string = event->string;

  if (event->type == CTX_KEY_DOWN)
  {
    ctx_queue_draw(ui->ctx);
    if (!strcmp(string, "right"))
    {
      widget->active = 1;
      widget->float_data =
      widget->float_data_target = widget->float_data + widget->step;
      if (widget->float_data_target >= widget->max_val)
        widget->float_data_target = widget->max_val;
    }
    else if (!strcmp(string, "left")) 
    {
      widget->active = 1;
  
      widget->float_data =
      widget->float_data_target = widget->float_data - widget->step;
      if (widget->float_data_target <= widget->min_val)
         widget->float_data_target = widget->min_val;
    }
    else if (!strcmp(string, "escape") || !strcmp(string, "space") ||
             !strcmp(string, "backspace") || !strcmp(string, "return"))
    {
      ui->active_id = NULL;
    }
    //printf("deactivated slider\n");
  }
  else if (event->type == CTX_KEY_UP)
  {
    ctx_queue_draw(ui->ctx);
    /* delayed deactivate is a hack to make it work with osk which 
     * has keydown and keyup in same frame
     */
    if (!strcmp(string, "right") || !strcmp(string, "left")) {
      ctx_add_timeout (ui->ctx, 200, widget_unactivate, widget);
    }
    if (!strcmp(string, "up"))
    {
      ui->active_id = NULL;
      ui_do (ui, "focus-previous");
    }
    if (!strcmp(string, "down"))
    {
      ui->active_id = NULL;
      ui_do (ui, "focus-next");
    }
  }

}

static void
ui_entry_key_press (CtxEvent *event, void *a,
                    void *b)
{
  Ui *ui = a;
  UiWidget *widget = b;
  const char *string = event->string;

  if (!strcmp(string, "space"))
    string = " ";
  if (!strcmp(string, "backspace")) {
    if (ui->cursor_pos) {
      int old_cursor_pos = ui->cursor_pos;
      ui->cursor_pos--;
      while ((ui->temp_text[ui->cursor_pos] & 192) == 128)
        ui->cursor_pos--;
      if (ui->cursor_pos < 0)
        ui->cursor_pos = 0;
      memmove(&ui->temp_text[ui->cursor_pos], &ui->temp_text[old_cursor_pos],
              strlen(&ui->temp_text[old_cursor_pos]) + 1);
    }
  } else if (!strcmp(string, "delete")) {
    if (ui->cursor_pos < (int)strlen(ui->temp_text)) {
      memmove(&ui->temp_text[ui->cursor_pos],
              &ui->temp_text[ui->cursor_pos + 1],
              strlen(&ui->temp_text[ui->cursor_pos + 1]) + 1);
      while ((ui->temp_text[ui->cursor_pos] & 192) == 128)
        memmove(&ui->temp_text[ui->cursor_pos],
                &ui->temp_text[ui->cursor_pos + 1],
                strlen(&ui->temp_text[ui->cursor_pos + 1]) + 1);
    }
  } else if (!strcmp(string, "return")) {
    widget->state = ui_state_commit;
    ui->active_id = NULL;
  } else if (!strcmp(string, "escape")) {
    ui->active_id = NULL;
    ui_do(ui, "kb-hide");
  } else if (!strcmp(string, "left")) {
    ui->cursor_pos--;
    while ((ui->temp_text[ui->cursor_pos] & 192) == 128)
      ui->cursor_pos--;
    if (ui->cursor_pos < 0)
      ui->cursor_pos = 0;
  } else if (!strcmp(string, "home")) {
    ui->cursor_pos = 0;
  } else if (!strcmp(string, "end")) {
    ui->cursor_pos = strlen(ui->temp_text);
  } else if (!strcmp(string, "right")) {
    ui->cursor_pos++;
    while ((ui->temp_text[ui->cursor_pos] & 192) == 128)
      ui->cursor_pos++;
    if ((int)strlen(ui->temp_text) < ui->cursor_pos)
      ui->cursor_pos = strlen(ui->temp_text);
  } 
  ctx_queue_draw(ui->ctx);
}

void ui_entry_activate (Ui *ui, void *id)
{
  UiWidget *widget = ui_widget_by_id(ui, id);
  if (ui->active_id != widget->id)
  {
    ui->active_id = widget->id;
    ui->cursor_pos = ctx_utf8_strlen (ui->temp_text);
  }
  ui->activate = 0;
}

static void ui_entry_text_input (CtxEvent *event, void *userdata,
                                 void *userdata2)
{
  Ui *ui = userdata;
  UiWidget *widget = userdata2;
  const char *string = event->string;
  int insert_len = strlen(string);
  ui_entry_activate (ui, widget->id);

  if (strlen(ui->temp_text) + insert_len + 1 < sizeof(ui->temp_text)) {
    memmove(&ui->temp_text[ui->cursor_pos + insert_len],
            &ui->temp_text[ui->cursor_pos],
            strlen(&ui->temp_text[ui->cursor_pos]) + 1);
    memcpy(&ui->temp_text[ui->cursor_pos], string, insert_len);
    ui->cursor_pos += insert_len;
  }
  ctx_queue_draw(ui->ctx);
}

float ui_get_font_size(Ui *ui) { return ui->font_size_px; }

static void ui_slider_drag_float(CtxEvent *event, void *data1, void *data2) {
  Ui *ui = data1;
  UiWidget *widget = ui_widget_by_id(ui, data2);
  ui->active_id = NULL;
  if (!widget)
    return;

  if (event->type == CTX_DRAG_PRESS)
  {
    widget->active = 1;
    ui->focused_id = widget->id;
  }
  else if (event->type == CTX_DRAG_RELEASE)
  {
    widget->active = 0;
  }

  float new_val = ((event->x - widget->x) / widget->width) *
                      (widget->max_val - widget->min_val) +
                  widget->min_val;
  if (new_val < widget->min_val)
    new_val = widget->min_val;
  if (new_val > widget->max_val)
    new_val = widget->max_val;
  widget->float_data = new_val;
  widget->float_data_target = new_val;
  event->stop_propagate = 1;
  ctx_queue_draw(ui->ctx);
}
void ui_start_overlay (Ui *ui, float fade_bg);
void ui_end_overlay (Ui *ui);

void ui_choice_draw (Ui *ui, float x, float y)
{
  UiWidget *widget = &ui->widgets[ui->widget_count-1];

  int chosen_val = (int)widget->float_data;
  int chosen_no = 0;
  for (int i = 0; i < ui->n_choices; i++)
    if (ui->choice_values[i] == chosen_val)
      chosen_no = i;

  ctx_save (ui->ctx);

  float width = em * 3;

  for (int i = 0; i < ui->n_choices; i++)
  {
    float owidth = ctx_text_width (ui->ctx, ui->choice_names[i]);
    if (owidth > width)
      width = owidth;
  }
  width += em ;
  ui_set_dim (ui, x, y, width, em *124);


  for (int i = 0; i < ui->n_choices; i++)
  {
    int flags = UI_SQUARE | UI_LEFT_ALIGN | UI_NO_BASE;
    STYLE_TEXT;
    if (i == chosen_no)
    {
      flags |= UI_REVERSE;
      flags -= UI_NO_BASE;
      if (widget->active == 2)
      {
        ui->focused_id = UI_ID_STR(ui->choice_names[i]);
        ctx_queue_draw (ui->ectx);
      }
    }
    ui_flags (ui, flags);
    if (ui_button (ui, ui->choice_names[i]))
    {
      widget->active = 0;
      widget->float_data = ui->choice_values[i];
    }
  }

  ctx_restore (ui->ctx);
}

static void ui_choice_clear (Ui *ui)
{
  if (ui->n_choices)
  {
    for (int i = 0; i < ui->n_choices; i++)
      if (ui->choice_names[i])
      {
        ctx_free (ui->choice_names[i]);
        ui->choice_names[i] = NULL;
      }
    if (ui->choice_name)
    {
      ctx_free (ui->choice_name);
      ui->choice_name = NULL;
    }
    ui->n_choices = 0;
  }
}

static int ui_choice_flush (Ui *ui)
{
  UiWidget *widget = &ui->widgets[ui->widget_count-1];
  if (widget->type == ui_type_choice && ui->n_choices)
  {
    ctx_save (ui->ctx);

    int chosen_val = (int)widget->float_data;
    int chosen_no = 0;
    for (int i = 0; i < ui->n_choices; i++)
      if (ui->choice_values[i] == chosen_val)
        chosen_no = i;


    float choice_width = 10 * em;
    char *tmp = NULL;
    char *s = ui->choice_names[chosen_no];
    if (strchr (s, '\b'))
    {
      s = tmp = ctx_strdup (s);
      strchr (tmp, '\b')[0] = 0;
    }
    choice_width = ctx_text_width (ui->ctx, s);
    ctx_reset_path (ui->ctx);
    ctx_round_rectangle (ui->ctx, ui->choice_x - em * 0.5, widget->y, choice_width + em * 2, em+em/4, em/2);

    ctx_save (ui->ctx);
    if (widget->hovered)
      STYLE_WIDGET_BASE_HOVERED;
    else
      STYLE_WIDGET_BASE;
    ctx_fill (ui->ctx);
    ctx_restore (ui->ctx);

    STYLE_TEXT;
    ctx_move_to (ui->ctx, ui->choice_x, widget->y + em * 1.0);
    ctx_text (ui->ctx, s);
    if (tmp)
      ctx_free (tmp);


    bool focused = (widget->id == ui->focused_id);

    if (focused)
    { 
      STYLE_WIDGET_BORDER_FOCUSED;
      ctx_line_width (ui->ctx, em * 0.3);
    }
    else
    {
      STYLE_WIDGET_BORDER;
    }
    ctx_round_rectangle (ui->ctx, ui->choice_x - em * 0.5, widget->y, choice_width + em * 2, em+em/4, em/2);
    ctx_stroke (ui->ctx);

    ctx_line_width (ui->ctx, em * 0.1);
    ctx_move_to (ui->ctx, ui->choice_x + choice_width + em * 0.4, widget->y + em * 0.4);
    ctx_line_to(ui->ctx, ui->choice_x + choice_width + em * 0.7, widget->y + em * 0.9);
    ctx_line_to(ui->ctx, ui->choice_x + choice_width + em * 1.0, widget->y + em * 0.4);
    ctx_stroke (ui->ctx);

    if (widget->active)
    {
      ui_start_overlay (ui, 1.0);

      float x = ui->choice_x;
      float y = widget->y;
      float width = em * 3;
      for (int i = 0; i < ui->n_choices; i++)
      {
        float owidth = ctx_text_width (ui->ctx, ui->choice_names[i]);
        if (owidth > width)
          width = owidth;
      }
      float height = em + ui->line_height * ui->n_choices;

     if (y + height > ctx_height (ui->ectx))
     {
       y = ctx_height (ui->ectx) - height;
       if (y < em * 0.5f)
         y = em * 0.5f;
     }
     if (x + width + em * 2> ctx_width (ui->ectx))
     {
       x = ctx_width (ui->ectx) - width - em * 2;
       if (x < em * 0.5f)
         x = em * 0.5f;
     }


    if (!(widget->flags & UI_NO_BASE))
    {

      ctx_save (ui->ctx);

     width += em ;


    ctx_round_rectangle (ui->ctx, x- em *.5, y- em *.5,width + em, height, em);
    STYLE_WIDGET_BASE;
    ctx_fill (ui->ctx);
    ctx_round_rectangle (ui->ctx, x- em *.5, y- em *.5,width + em, height, em);
    STYLE_WIDGET_BORDER;
    ctx_stroke (ui->ctx);
      ctx_restore (ui->ctx);
    }

      widget->active ++;
      ui_choice_draw (ui, x, y);

      ui_end_overlay (ui);
    }
    ctx_restore (ui->ctx);
  }
  return 0;
}

static UiWidget *ui_widget_register(Ui *ui, ui_type type,
                                    float x, float y,
                                    float width, float height, void *id) {
  UiFlag flags = ui->flags;
  ui->flags = 0;
  if (ui->widget_count + 1 >= UI_MAX_WIDGETS) {
    printf("too many widgets\n");
    return &ui->widgets[ui->widget_count];
  }

  UiWidget *widget = &ui->widgets[ui->widget_count++];
  if (widget->id != id) { // if new widgets are insterted our state is reset
    widget->id = id;
    widget->state = ui_state_default;
    widget->fresh = 1;
    widget->type = type;
    widget->overlay_no = ui->overlay_no;
  } else {
    widget->fresh = 0;
  }
  widget->flags = flags;
  widget->x = x;
  widget->y = y;
  widget->width = width;
  widget->height = height;

  widget->hovered = (ui->pointer_x >= widget->x &&
      ui->pointer_x < widget->x + widget->width &&
      ui->pointer_y >= widget->y &&
      ui->pointer_y < widget->y + widget->height);
  if (widget->overlay_no != ui->prev_max_overlay)
  {
    widget->hovered = 0; // we cannot fix the ones before though
                         // perhaps with a 1 frame delay
  }

  widget->visible = 1;
      //(x >= -em && x < ui->width + em && y >= -em && y < ui->height + em);

  ui->last_id = widget->id;

  return widget;
}

void ui_focus (Ui *ui, void *id)
{
  ui->focused_id = id;
}

float ui_slider_coords(Ui *ui, void *id, float x, float y, float width,
                       float height, float min_val, float max_val, float step,
                       float value)
{
  Ctx *ctx = ui->ctx;
  Ctx *ectx = ui->ectx;
  UiWidget *widget =
      ui_widget_register(ui, ui_type_slider,
                         x, y-em*0.3, width, height+em*0.1, id);
  if (widget->fresh || !widget->active) {
    widget->float_data = value;
    widget->float_data_target = value;
  }
  widget->min_val = min_val;
  widget->max_val = max_val;
  widget->step = step;

  bool focused = (widget->id == ui->focused_id);
  if (focused && ui->activate) {
    widget->float_data = value;
    widget->float_data_target = value;
    ui->active_id = widget->id;
    ui->activate = 0;
  }

  float rel_value =
      ((value)-widget->min_val) / (widget->max_val - widget->min_val);
  if (ui->active_id == widget->id) {
  //  value = widget->float_data =
  //      (value * 0.8 + 0.2 * widget->float_data_target);
    value = widget->float_data = widget->float_data_target;
  }

  if (widget->visible) {

    ctx_save(ctx);

    STYLE_WIDGET_BORDER;
    ctx_move_to(ctx, x, y + height / 2);
    ctx_line_to(ctx, x + width, y + height / 2);
    ctx_stroke(ctx);
    int hovered = ( (widget->hovered && (ctx_fabsf(ui->pointer_x - (x+ rel_value *width)) < height)));

    if (focused)
      STYLE_WIDGET_BORDER_FOCUSED;
    else
      STYLE_WIDGET_BORDER;

    float gap = 0.06;
    if (focused)
      gap *= 2;

    float radius_1 = 0.35;

    if ( hovered || widget->active || (ui->active_id == widget->id))
    {
      radius_1 += 0.15;
    }
    float radius_0 = radius_1 + gap;

    ctx_arc(ctx, x + rel_value * width, y + height / 2, height * radius_0, 0,
            2 * 3.1415, 0);
    ctx_fill(ctx);

    if (widget->id == UI_ID_STR("cool-hue "))
      ui_color (ui, UI_COLOR_FULL, 0.5, -1, 1);
    else if (widget->id == UI_ID_STR("warm-hue "))
      ui_color (ui, UI_COLOR_FULL, 0.5, 1, 1);
    else if (ui->active_id == widget->id)
      STYLE_WIDGET_BASE_FOCUSED;
    else if (hovered)
      STYLE_WIDGET_BASE_HOVERED;
    else
      STYLE_WIDGET_BASE;

    ctx_arc(ctx, x + rel_value * width, y + height / 2, height * radius_1, 0.0,
            3.1415 * 2, 0);
    ctx_fill(ctx);

    if (ui->overlay_no == ui->max_overlay)
    {
      ctx_rectangle(ectx, x + rel_value * width - height * 0.75, y, height * 1.5,
                    height);
      ctx_listen(ectx, CTX_DRAG, ui_slider_drag_float, ui, widget->id);
      ctx_listen_set_cursor (ctx, CTX_CURSOR_HAND);
      ctx_reset_path(ectx);
    }

    ctx_restore(ctx);
  }

  return widget->float_data;
}



static void ui_choice_drag(CtxEvent *event, void *data1, void *data2) {
  Ui *ui = data1;
  UiWidget *widget = ui_widget_by_id(ui, data2);
  if (!widget)
    return;
  if (event->type == CTX_DRAG_PRESS) {
    widget->active = 1;
    //ui->active_id = widget->id;
    ui->focused_id = widget->id;
    ctx_queue_draw (event->ctx);
  } else if (event->type == CTX_DRAG_RELEASE) {
  } else
  {
  }
}
static void ui_color_popup_drag (CtxEvent *event, void *a, void *b)
{
  ui_choice_drag (event, a, b);
}

static void ui_button_drag(CtxEvent *event, void *data1, void *data2) {
  Ui *ui = data1;
  UiWidget *widget = ui_widget_by_id(ui, data2);
  if (!widget)
    return;

  if (event->type == CTX_DRAG_PRESS) {
    ui_set_focus(ui, widget);
    widget->state = ui_state_hot;
  } else if (event->type == CTX_DRAG_RELEASE) {
    if (widget->id == ui->focused_id) {
      if (widget->state == ui_state_hot)
        ui->activate = 1;
    }
  } else {
    if ((event->y < widget->y) || (event->x < widget->x) ||
        (event->y > widget->y + widget->height) ||
        (event->x > widget->x + widget->width)) {
      widget->state = ui_state_lost_focus;
    } else {
      widget->state = ui_state_hot;
    }
  }
  ctx_queue_draw(ui->ctx);
  event->stop_propagate = 1;
}



int ui_button_coords(Ui *ui, float x, float y, float width, float height,
                     const char *label, void *id)
{
  UiFlag flags = ui->flags;
  Ctx *ctx = ui->ctx;
  Ctx *ectx = ui->ectx;
  if (width <= 0)
    width = ctx_text_width(ctx, label);
  if (height <= 0)
    height = em * 1.4;

  UiWidget *widget =
      ui_widget_register(ui, ui_type_button, x, y, width, height, id);

  bool focused = (widget->id == ui->focused_id);

  if (widget->visible ) {
    ctx_reset_path (ctx);
    ctx_save(ctx);

    STYLE_TEXT;

    if (flags & UI_LEFT_ALIGN)
      ctx_text_align(ctx, CTX_TEXT_ALIGN_START);
    else
      ctx_text_align(ctx, CTX_TEXT_ALIGN_CENTER);

    float radius =
      ((flags & UI_SQUARE)==0)?em * 0.5f:em * 0.01f;

    if ((flags & UI_NO_BASE) && !focused)
    {

    }
    else
    {
      if ((flags & UI_NO_BASE) == 0)
      {
        ctx_round_rectangle(ctx, x, y, width, height, radius);
        ctx_save (ctx);
        if (flags & UI_REVERSE)
        {
          if (widget->hovered)
            STYLE_HOVERED;
          else
            STYLE_TEXT;
        }
        else
        {
          if (widget->hovered)
            STYLE_WIDGET_BASE_HOVERED;
          else
            STYLE_WIDGET_BASE;
        }
        ctx_fill(ctx);
        ctx_restore (ctx);
      }

      ctx_save (ctx);
      if (flags & UI_REVERSE)
      {
        if (focused)
        {
          STYLE_WIDGET_BASE_FOCUSED;
          ctx_line_width (ctx, em * 0.3);
        }
        else
          STYLE_WIDGET_BASE;
      }
      else
      {
        if (focused)
        {

          STYLE_FOCUSED;
          ctx_line_width (ctx, em * 0.3);
        }
        else
          STYLE_WIDGET_BORDER;
      }

      if (focused || ((flags & UI_NO_BASE)==0))
      {
        ctx_round_rectangle(ctx, x, y, width, height, radius);
        ctx_close_path (ctx); // XXX: should not be needed?
        ctx_stroke (ctx);
      }
      ctx_restore (ctx);

    }

    if (flags & UI_REVERSE)
      STYLE_WIDGET_BASE;

    if (flags & UI_LEFT_ALIGN)
      ctx_move_to(ctx, x + em, y + em);
    else
      ctx_move_to(ctx, x + width / 2, y + em);

    if(0) //if (strchr (label, '\b'))
    {
      char *tmp = ctx_strdup (label);
      strchr (tmp, '\b')[0] = 0;
      ctx_text(ctx, tmp);
      ctx_free (tmp);
    }
    else
    {
      ctx_text(ctx, label);
    }

    ctx_reset_path(ectx);
    if (ui->overlay_no == ui->max_overlay)
    {
      ctx_reset_path(ectx);
      ctx_rectangle(ectx, x, y, width, height);
      ctx_listen(ectx, CTX_DRAG, ui_button_drag, ui, widget->id);
      ctx_listen_set_cursor (ectx, CTX_CURSOR_HAND);
      ctx_reset_path(ectx);
    }
    ctx_restore(ctx);
  }

  if ((focused && ui->activate)) {
    ui->activate = 0;
    widget->state = ui_state_default;
    ctx_queue_draw(ectx);
    return 1;
  }
  return 0;
}

static int lazy_next (Ctx *ctx, void *a)
{
  Ui *ui = a;    ui_do (ui, "focus-next");
  return 0;
}

static int lazy_prev (Ctx *ctx, void *a)
{
  Ui *ui = a;    ui_do (ui, "focus-previous");
                 ui_do (ui, "focus-previous");
  return 0;
}

void ui_draw_overlays (Ui *ui)
{
  for (int i = 1; i <= ui->max_overlay; i++)
  {
    ctx_save (ui->ectx);
    ctx_render_ctx (ui->overlay[i].drawlist, ui->ectx);
    ctx_restore (ui->ectx);
  }
  ui->drawn_overlays++;
}

static int has_prefix (const char *str, const char *needle)
{
  if (!strncmp (str, needle, strlen (needle)))
    return 1;
  return 0;
}

enum {
  COLOR_TYPE_GRAY4,
  COLOR_TYPE_GRAY,
  COLOR_TYPE_RGB4,
  COLOR_TYPE_RGBA4,
  COLOR_TYPE_RGB,
  COLOR_TYPE_RGBA,
  COLOR_TYPE_NAME
};

static float decode_hex (int a, int b)
{
  if (a >= 'a' && a <= 'f')
    a=a-'a'+10;
  else if (a >= 'A' && a <= 'F')
    a=a-'A'+10;
  else if (a >= '0' && a <= '9')
    a=a-'0';

  if (b >= 'a' && b <= 'f')
    b=b-'a'+10;
  else if (b >= 'A' && b <= 'F')
    b=b-'A'+10;
  else if (b >= '0' && b <= '9')
    b=b-'0';

  return (a * 16 + b)/255.0f;
}


void str_to_color (const char *str, float *rgba)
{
  rgba[3]=1.0f;

  if (str[0]=='#')
    {
      int len = strlen (str);
      switch (len - 1)
      {
        case 1:
          rgba[0]=
          rgba[1]=
          rgba[2]= decode_hex (str[1], str[1]);
          break;
        case 2:
          rgba[0]=
          rgba[1]=
          rgba[2]= decode_hex (str[1], str[2]);
          break;
        case 3:
          rgba[0]= decode_hex (str[1], str[1]);
          rgba[1]= decode_hex (str[2], str[2]);
          rgba[2]= decode_hex (str[2], str[2]);
          break;
        case 4:
          rgba[0]= decode_hex (str[1], str[1]);
          rgba[1]= decode_hex (str[2], str[2]);
          rgba[2]= decode_hex (str[2], str[2]);
          rgba[3]= decode_hex (str[3], str[3]);
          break;
        case 6:
          rgba[0]= decode_hex (str[1], str[2]);
          rgba[1]= decode_hex (str[3], str[4]);
          rgba[2]= decode_hex (str[5], str[6]);
          break;
        case 8:
          rgba[0]= decode_hex (str[1], str[2]);
          rgba[1]= decode_hex (str[3], str[4]);
          rgba[2]= decode_hex (str[5], str[6]);
          rgba[3]= decode_hex (str[7], str[8]);
          break;
        default:
          rgba[0]=rgba[1]=rgba[2]=rgba[3]=0.0;
          break;
    }
  }
  else
  {
    rgba[0]=rgba[1]=rgba[2]=rgba[3]=0.0;
  }
}

static void ui_incoming_message (CtxEvent *e, void *a, void *b)
{
  Ui *ui = a;
  const char *msg = e->string;
  float rgba[4];

  if (has_prefix (msg, "ui-brightness:"))
    ui_set_brightness (ui, atof (strchr(msg, ':')+1));
  else if (has_prefix (msg, "ui-contrast:"))
    ui_set_contrast (ui, atof (strchr(msg, ':')+1));
  else if (has_prefix (msg, "ui-saturation:"))
    ui_set_saturation (ui, atof (strchr(msg, ':')+1));
  else if (has_prefix (msg, "ui-base-saturation:"))
    ui_set_base_saturation (ui, atof (strchr(msg, ':')+1));
  else if (has_prefix (msg, "ui-brightest:"))
  {
    str_to_color (strchr(msg, ':')+1, rgba);
    ui_color_set_brightest (ui, rgba);
  }
  else if (has_prefix (msg, "ui-darkest:"))
  {
    str_to_color (strchr(msg, ':')+1, rgba);
    ui_color_set_darkest (ui, rgba);
  }
  else if (has_prefix (msg, "ui-cool:"))
  {
    str_to_color (strchr(msg, ':')+1, rgba);
    ui_color_set_cool (ui, rgba);
  }
  if (has_prefix (msg, "ui-warm:"))
  {
    str_to_color (strchr(msg, ':')+1, rgba);
    ui_color_set_warm (ui, rgba);
  }


  ctx_queue_draw (e->ctx);
}

void ui_end_frame(Ui *ui) {
  if (!ui->drawn_overlays)
    ui_draw_overlays (ui);
  ui->drawn_overlays = 0;
  for (int i = 1; i <= ui->max_overlay; i++)
  {
    ctx_destroy (ui->overlay[i].drawlist);
    ui->overlay[i].drawlist = NULL;
  }

  Ctx *ctx = ui->ctx = ui->ectx;
  

  if (ui->active_id) {
    UiWidget *widget = ui_widget_by_id(ui, ui->active_id);
    if (widget)
      switch (widget->type) {
      case ui_type_button:
      case ui_type_colorpopup:
      case ui_type_choice:
      case ui_type_none:
        break;
      case ui_type_entry:
        ctx_listen(ctx, CTX_KEY_PRESS, ui_entry_key_press, ui, widget);
        ctx_listen(ctx, CTX_TEXT_INPUT, ui_entry_text_input, ui, widget);
        break;
      case ui_type_slider:
        ctx_listen(ctx, CTX_KEY_DOWN|CTX_KEY_UP, ui_slider_key_press, ui, widget);
        break;
      }
  } else {

    UiWidget *fwidget = ui_widget_by_id(ui, ui->focused_id);
    if (fwidget)
    {
      ctx_listen(ctx, CTX_TEXT_INPUT, ui_entry_text_input, ui, fwidget);
    }

    if (0) // 4way - nav, make it a toggle?
    {
      ui_add_key_binding(ui, "up", "focus-up", "previous focusable item");
      ui_add_key_binding(ui, "left", "decrease-or-focus-left", "previous focusable item");
      ui_add_key_binding(ui, "down", "focus-down", "next focusable item");
      ui_add_key_binding(ui, "right", "increase-or-focus-right", "next focusable item");
    }
    else
    {
      ui_add_key_binding(ui, "up", "focus-previous", "previous focusable item");
      ui_add_key_binding(ui, "left", "decrease-or-focus-previous", "previous focusable item");
      ui_add_key_binding(ui, "down", "focus-next", "next focusable item");
      ui_add_key_binding(ui, "right", "increase-or-focus-next", "next focusable item");

    }

    ui_add_key_binding(ui, "shift-tab", "focus-previous",
                       "previous focusable item");



    ui_add_key_binding(ui, "tab", "focus-next", "next focusable item");
    ui_add_key_binding(ui, "space", "activate", "activate");
    ui_add_key_binding(ui, "return", "activate", "activate");
  }
  ui_add_key_binding(ui, "control-q", "exit", "quit");

  if (ui->max_overlay)
    ui_add_key_binding(ui, "escape", "close-overlay", "quit");
  ctx_restore (ctx);
#if 1
  if (ui->prev_max_overlay > ui->max_overlay)
  {
    ui->focused_id = ui->overlay[ui->max_overlay].focused_id;
    ctx_queue_draw (ui->ctx);
  }
#endif
  ui->prev_max_overlay = ui->max_overlay;

#if 0 // uncomment for report of widget count high watermark
  static int max_widgets = 0;
  if (ui->widget_count > max_widgets)
  {
    max_widgets = ui->widget_count;
    fprintf (stderr, "max w:%i\r", max_widgets);
  }
#endif

  ctx_reset_path (ctx);
  ctx_listen(ctx, CTX_MESSAGE, ui_incoming_message, ui, NULL);
}

void
ui_start_frame(Ui *ui)
{
  Ctx *ctx = ui->ctx;
  ui->last_id = NULL;
  ui->overlay_no = 0;
  ui->max_overlay = 0;
  ctx_save (ctx);

  static long int prev_ticks = -1;
  long int ticks = ctx_ticks();
  long int ticks_delta = ticks - prev_ticks;
  if (ticks_delta > 1000000 || prev_ticks == -1)
    ticks_delta = 10;
  prev_ticks = ticks;
  ui->delta_ms = ticks_delta / 1000;
  ui->font_size_px = ctx_get_font_size (ctx);

  ui->width  = ctx_width(ctx);
  ui->height = ctx_height(ctx);
  ui->line_height = ui->font_size_px * 1.5;
  ctx_rectangle(ctx, 0, 0, ui->width, ui->line_height * 100);//ui->height);
  ctx_reset_path(ctx);
  //ctx_text_align(ctx, CTX_TEXT_ALIGN_CENTER);
  ctx_text_align(ctx, CTX_TEXT_ALIGN_START);
  //ctx_font_size(ctx, ui_get_font_size(ui));
  ctx_font(ctx, "Regular");


  ui->y = (int)(ui->height * 0.15);
  ui->widget_count = 0;

  ui_color_ink (ui, 1.0, 0.0);
  ctx_line_width (ctx, em * 0.1);

  ui->pointer_x_global = ctx_pointer_x (ctx);
  ui->pointer_y_global = ctx_pointer_y (ctx);

  ui_set_dim (ui, em/2, em/2, ui->width-em,ui->height-em);
}

// sets the dimensions that we are doing layout within
//    width provides width of this vertical stretch of auto-layed out widgets
void ctx_device_to_user          (Ctx *ctx, float *x, float *y);

void ui_set_dim(Ui *ui, float x0, float y0, float width, float height)
{
  ui->x0 = x0;
  ui->y0 = y0;
  ui->x = x0;
  ui->y = y0;
  ui->width = width;
  ui->height = height;

  ui->pointer_x = ui->pointer_x_global;
  ui->pointer_y = ui->pointer_y_global;
  ctx_device_to_user (ui->ctx, &ui->pointer_x, &ui->pointer_y);
}

void ui_y_advance (Ui *ui, float amount)
{
  ui->y += amount;
  ui->x = ui->x0;
}

float
ui_slider (Ui *ui, const char *label,
           float min, float max, float step,
           float value)
{
  int len = strlen (label);
  char buf[32+len];
  Ctx *ctx = ui->ctx;
  ctx_save(ctx);
  STYLE_TEXT;

  sprintf (buf, "%s ", label);
  void *id = UI_ID_STR(buf);
  void *entry_id = UI_ID_STR(label);

 // UiWidget *entry = ui_widget_by_id (ui, entry_id);

  sprintf (buf, "%s  %.2f", label, value);

  int skip_slider = 0;
  float ret = 0.0;


  if (!(ui->flags & UI_NO_LABEL))
  {

  if (ui->edit_slider == id)
  {
    sprintf (buf, "%.2f", value);
    char *foo = ui_entry (ui, label, "", buf);

    if (foo)
    {
       ret = ctx_atof (foo);
       ui->edit_slider = NULL;
       ctx_free (foo);
       ui->focused_id = id;
       skip_slider = 1;
    }
    else
    if (ui->focused_id != entry_id)
    {
      ui->focused_id = entry_id;
      ctx_queue_draw (ui->ectx);
    }
    else
    {
      ui_entry_activate (ui, entry_id);
      ctx_queue_draw (ui->ectx);
    }

    ui->y -= ui->line_height * 0.4;
  }
  else
  {
    STYLE_TEXT;//ctx_rgba (ui->ctx, 0.5, 1, 0.5, 1);
    ui_text (ui, buf);
  }
  ui->y -= em * 0.5f;
  }


  if (skip_slider)
  {
    // not really skipping it, but dropping its result
    ui_slider_coords(ui, id, ui->x0, ui->y,
                     ui->width - em, ui->line_height, min, max, step, value);
  }
  else
  {
    ret = ui_slider_coords(ui, id, ui->x0, ui->y,
                           ui->width - em, ui->line_height, min, max, step, value);
  }

  ui_y_advance (ui, ui->line_height);

  ctx_restore(ctx);
  return ret;
}

bool ui_toggle (Ui *ui, const char *label, bool value)
{
  Ctx *ctx = ui->ctx;
  ctx_save (ctx);
  ui_flags (ui, UI_SQUARE);
  if (ui_button_coords(ui, ui->x0 + ui->width - em * 2.0, ui->y, em * 1.3, em * 1.3, value==0?" ":"V", UI_ID_STR(label)))
    value = !value;
  ctx_restore (ctx);

  if (ui->y > -2 * em && ui->y < ui->height - em)
  {
    STYLE_TEXT;
    ctx_move_to(ctx, ui->x0, ui->y + em);
    ctx_save (ctx);
    ctx_wrap_left (ctx, ui->x0);
    ctx_wrap_right (ctx, ui->x0 + ui->width - em * 2);
    ctx_text(ctx, label);
    ui->y = ctx_y (ctx);
    ctx_restore (ctx);
  }

  ui_y_advance (ui, ui->line_height - em);
  return value;
}

void ui_title(Ui *ui, const char *string) {
  Ctx *ctx = ui->ctx;
  float scale = 1.4f;
  STYLE_TEXT;
  /* if (ui->y > -em && ui->y < ui->height + em) */ {
    ctx_save(ctx);
    ctx_font_size(ctx, ctx_get_font_size(ctx) * scale);
    ctx_move_to(ctx, ui->x0, ui->y + em * scale);
    ctx_text(ctx, string);
    ctx_restore(ctx);
  };
  ui_y_advance (ui, ui->line_height + em * scale - em);
}

void ui_seperator(Ui *ui)
{
  Ctx *ctx = ui->ctx;
  ctx_move_to(ctx, ui->width * 0.4, ui->y + ui->line_height * 0.3);
  ctx_rel_quad_to(ctx, ui->width * 0.2, -ui->line_height * 0.1, ui->width * 0.2,
                  -ui->line_height * 0.3);
  ctx_rel_quad_to(ctx, -ui->width * 0.1, 0.0, -ui->width * 0.2,
                  ui->line_height * 0.7);
  ctx_rel_quad_to(ctx, 0, -ui->line_height * 0.3, ui->width * 0.2,
                  -ui->line_height * 0.3);
  ui_color_ink (ui, 0.5, 0);
  ctx_stroke(ctx);
  ui_y_advance (ui, ui->line_height);
}

void ui_text (Ui *ui, const char *string)
{
  Ctx *ctx = ui->ctx;
  /* if (ui->y > -em && ui->y < ui->height + em) */ {
    ctx_save (ctx);
    ctx_wrap_left (ctx, ui->x0);
    ctx_wrap_right (ctx, ui->x0 + ui->width);
    ctx_move_to(ctx, ui->x0, ui->y + em);
    //STYLE_TEXT; XXX
    ui_color_ink (ui, 1.0, 0.0);
    ctx_text(ctx, string);
    ctx_restore (ctx);
  };

  for (int i = 0; string[i]; i++)
  {
    if (string[i] == '\n')
      ui_y_advance (ui, ui->line_height);
  }
  ui_y_advance (ui, ui->line_height);
}

void ui_textf (Ui *ui, const char *format, ...)
{
  va_list ap;
  size_t needed;
  char *buffer;
  va_start (ap, format);
  needed = vsnprintf (NULL, 0, format, ap) + 1;
  buffer = malloc (needed);
  va_end (ap);
  va_start (ap, format);
  vsnprintf (buffer, needed, format, ap);
  va_end (ap);
  ui_text (ui, buffer);
  free (buffer);
}

int ui_button (Ui *ui, const char *label)
{
  UiFlag flags = ui->flags;
  if (flags & UI_INLINE)
  {
    float width = ctx_text_width (ui->ctx, label);
    width += em;
    ui->x += width + em * 0.5;
    if (ui->x > ui->x0 + ui->width)
    {
      ui->x = ui->x0 + width + em * 0.5;
      ui->y += ui->line_height;
    }
  
    return 
           ui_button_coords (ui, ui->x - width - em * 0.5, ui->y,
                             width, ui->line_height * 0.8, label,
                             UI_ID_STR(label));
  }
  ui_y_advance (ui, ui->line_height);
  return 
         ui_button_coords (ui, ui->x, ui->y - ui->line_height,
                           ui->width, ui->line_height * 0.8, label,
                           UI_ID_STR(label));
}

void ui_newline (Ui *ui)
{
  ui_y_advance (ui, ui->line_height);
}

static void ui_entry_drag(CtxEvent *event, void *data1, void *data2) {
  Ui *ui = data1;
  UiWidget *widget = ui_widget_by_id(ui, data2);
  if (!widget)
    return;

  if (event->type == CTX_DRAG_PRESS) {
    event->stop_propagate = 0;
    ui_set_focus(ui, widget);
    widget->state = ui_state_default;
  } else if (event->type == CTX_DRAG_RELEASE) {
    event->stop_propagate = 0;
    if (widget->state != ui_state_lost_focus) {
      if (data2 != ui->active_id) {
        ui_do(ui, "kb-show");
        ui_do(ui, "activate");
      }
    }
  } else {
    if ((event->y < widget->y) || (event->x < widget->x) ||
        (event->y > widget->y + widget->height) ||
        (event->x > widget->x + widget->width)) {
      widget->state = ui_state_lost_focus;
    }
  }
}


char *ui_entry_coords(Ui *ui, void *id, float x, float y, float w, float h,
                      const char *fallback, const char *value) {
  Ctx *ctx = ui->ctx;
  Ctx *ectx = ui->ectx;
  UiWidget *widget = ui_widget_register(ui, ui_type_entry, x, y, w, h, id);

  bool focused = (id == ui->focused_id);

  STYLE_TEXT;

  //if (focused && ui->active_id == NULL) {
  if (focused && (ui->active_id == NULL) && widget->state != ui_state_commit) {
    if (value)
      strcpy(ui->temp_text, value);
    else {
      ui->temp_text[0] = 0;
    }
  }
  if (focused && ui->activate) {
    ui_entry_activate (ui, widget->id);
    ui->cursor_pos = strlen(ui->temp_text);
  }

  const char *to_show = value;
  if (ui->active_id == widget->id)
    to_show = &ui->temp_text[0];


  if (widget->visible) {
    ctx_save(ctx);

  if (!(to_show && to_show[0]))
  {
    to_show = fallback;
    ctx_global_alpha (ctx, 0.5f);
  }


    {
      ctx_save (ctx);
      ctx_rectangle (ctx, widget->x, widget->y, widget->width, widget->height);


      if (focused) 
        STYLE_WIDGET_BASE;//_FOCUSED;
      else if (widget->hovered)
        STYLE_WIDGET_BASE_HOVERED;
      else
        STYLE_WIDGET_BASE;


      ctx_fill (ctx);

      if (focused && (ui->active_id != widget->id) )
      {
        ctx_rectangle (ctx, widget->x, widget->y, widget->width, widget->height);
        STYLE_WIDGET_BORDER_FOCUSED;
        ctx_line_width (ctx, em * 0.2);
        ctx_stroke (ctx);

      }

      ctx_restore (ctx);
    }

    ctx_text_align(ctx, CTX_TEXT_ALIGN_START);
    if (ui->active_id != widget->id) {
      ctx_move_to(ctx, x, y + em);
      ctx_font_size(ctx, em);

      if (to_show && to_show[0]) {
        if (to_show == fallback)
          STYLE_TEXT;
        ctx_text(ctx, to_show);
      }
    } else {
      char temp = ui->temp_text[ui->cursor_pos];
      float tw_pre = 0;
      float tw_selection = 0;
      int sel_bytes = 0;

      ctx_font_size(ctx, em);
      ui->temp_text[ui->cursor_pos] = 0;
      tw_pre = ctx_text_width(ctx, ui->temp_text);
      ui->temp_text[ui->cursor_pos] = temp;
      if (ui->selection_length) {
        for (int i = 0; i < ui->selection_length; i++) {
          sel_bytes += ctx_utf8_len(ui->temp_text[ui->cursor_pos + sel_bytes]);
        }
        temp = ui->temp_text[ui->cursor_pos + sel_bytes];
        ui->temp_text[ui->cursor_pos + sel_bytes] = 0;
        tw_selection = ctx_text_width(ctx, &ui->temp_text[ui->cursor_pos]);
        ui->temp_text[ui->cursor_pos + sel_bytes] = temp;
      }

      ctx_move_to(ctx, x, y + em);

      if (to_show && to_show[0]) {
        if (to_show == fallback)
          STYLE_TEXT;
        ctx_text(ctx, to_show);
      }

      ctx_rectangle(ctx, x + tw_pre - 1, y + 0.1 * em, em * 0.2 + tw_selection,
                    em);
      ctx_save(ctx);
      STYLE_WIDGET_BORDER_FOCUSED; // XXX
      ctx_fill(ctx);
      ctx_restore(ctx);
    }

    if (ui->overlay_no == ui->max_overlay)
    {
      ctx_reset_path(ectx);
      ctx_rectangle(ectx, x, y, w, h);
      ctx_listen(ectx, CTX_DRAG, ui_entry_drag, ui, widget->id);
      ctx_listen_set_cursor (ectx, CTX_CURSOR_HAND);
      ctx_reset_path(ectx);
    }

    ctx_restore(ctx);
  }

  if (widget->state == ui_state_commit) {
    widget->state = ui_state_default;
    ui_do(ui, "kb-hide");
    return ctx_strdup(ui->temp_text);
  }
  return NULL;
}



char *ui_color_popup_coords(Ui *ui, void *id, float x, float y, float w, float h,
                      const char *value)
{
  Ctx *ctx = ui->ctx;
  Ctx *ectx = ui->ectx;
  UiWidget *widget = ui_widget_register(ui, ui_type_colorpopup, x, y, w, h, id);

  bool focused = (id == ui->focused_id);

      float rgba[4] = {0.0f, 0.0f, 0.0f, 1.0f};
      float rgba_new[4];
      const char *str = value;
      str_to_color (str, rgba);

      for (int i = 0; i<4;i++)rgba_new[i]=rgba[i];

  if (widget->visible) 
  {
    ctx_save(ctx);

    {
      ctx_save (ctx);
      ctx_rectangle (ctx, widget->x, widget->y, widget->width, widget->height);


      if (focused) 
        STYLE_WIDGET_BASE;//_FOCUSED;
      else if (widget->hovered)
        STYLE_WIDGET_BASE_HOVERED;
      else
        STYLE_WIDGET_BASE;


      ctx_fill (ctx);

      ctx_rectangle (ui->ctx, x, y, ui->line_height, ui->line_height );
      ctx_rectangle (ctx, widget->x, widget->y, widget->width, widget->height);
      ctx_rgba (ui->ctx, rgba[0], rgba[1], rgba[2], rgba[3]);
      ctx_fill (ui->ctx);


      if (focused && (ui->active_id != widget->id) )
      {
        ctx_rectangle (ctx, widget->x, widget->y, widget->width, widget->height);
        STYLE_WIDGET_BORDER_FOCUSED;
        ctx_line_width (ctx, em * 0.2);
        ctx_stroke (ctx);

      }

      ctx_restore (ctx);
    }

    if (ui->active_id != widget->id) {
      ctx_move_to(ctx, x, y + em);
    } else {
    }

    if (ui->overlay_no == ui->max_overlay)
    {
      ctx_reset_path(ectx);
      ctx_rectangle(ectx, x, y, w, h);
      ctx_listen(ectx, CTX_DRAG, ui_color_popup_drag, ui, widget->id);
      ctx_listen_set_cursor (ectx, CTX_CURSOR_HAND);
      ctx_reset_path(ectx);
    }

    ctx_restore(ctx);

  }
    if (widget->active)
    {
      if (widget->active == 1)
        strcpy (ui->backup_text, value);
      ui_start_overlay (ui, 1.0);

      float x = widget->x;
      float y = widget->y;
      float width = em * 10;
      float height = em * 14;

     if (y + height > ctx_height (ui->ectx))
     {
       y = ctx_height (ui->ectx) - height;
       if (y < em * 0.5f)
         y = em * 0.5f;
     }
     if (x + width + em * 2> ctx_width (ui->ectx))
     {
       x = ctx_width (ui->ectx) - width - em * 2;
       if (x < em * 0.5f)
         x = em * 0.5f;
     }


    if (!(widget->flags & UI_NO_BASE))
    {

      ctx_save (ui->ctx);

     width += em ;


    ctx_round_rectangle (ui->ctx, x- em *.5, y- em *.5,width + em, height, em);
    STYLE_WIDGET_BASE;
    ctx_fill (ui->ctx);
    ctx_round_rectangle (ui->ctx, x- em *.5, y- em *.5,width + em, height, em);
    STYLE_WIDGET_BORDER;
    ctx_stroke (ui->ctx);
      ctx_restore (ui->ctx);
    }

      widget->active ++;




      STYLE_TEXT;


      ctx_move_to (ui->ctx, x, y + em);
      ctx_text (ui->ctx, value);


      ctx_rectangle (ui->ctx, x + em * 6, y , em ,em );
      ctx_rgba (ui->ctx, 0,0,0,1);
      ctx_fill (ui->ctx);
      ctx_rectangle (ui->ctx, x + em * 7, y , em ,em );
      ctx_rgba (ui->ctx, 0.5,0.5,0.5,1);
      ctx_fill (ui->ctx);
      ctx_rectangle (ui->ctx, x + em * 8, y , em ,em );
      ctx_rgba (ui->ctx, 1,1,1,1);
      ctx_fill (ui->ctx);

      ctx_rectangle (ui->ctx, x + em * 6.4, y + em * 0.5 , em * 2 , em * 2);
      ctx_rgba (ui->ctx, rgba[0], rgba[1], rgba[2], rgba[3]);
      ctx_fill (ui->ctx);

      ui_set_dim (ui, x, y, width, height);

      ui_newline (ui);

      rgba_new[0] = ui_slider (ui, "red", 0.0, 1.0, 1/100.0, rgba[0]);
      rgba_new[1] = ui_slider (ui, "green", 0.0, 1.0, 1/100.0, rgba[1]);
      rgba_new[2] = ui_slider (ui, "blue", 0.0, 1.0, 1/100.0, rgba[2]);

      if ((widget->flags & UI_NO_ALPHA)==0)
        rgba_new[3] = ui_slider (ui, "alpha", 0.0, 1.0, 1/100.0, rgba[3]);

      STYLE_TEXT;

      ui_flags (ui, UI_INLINE);

      int in_reset =0;
      if (ui_button (ui, "ok"))
      {
        ui_do (ui, "close-overlay");
      }
      ui_flags (ui, UI_INLINE);
      if (ui_button (ui, "reset"))
      {
        ui_end_overlay (ui);
        ui_do (ui, "close-overlay");
        return ctx_strdup (ui->backup_text);
      }


      if ((! in_reset) && memcmp(rgba, rgba_new, sizeof (rgba)))
      {
        const char hexdigits[17]="0123456789abcdef";
        ui->temp_text[0]='#';
        for (int c = 0; c < 4; c ++)
        {
          int val = rgba_new[c] * 255.0f;
          if (val < 0) val = 0;
          if (val > 255) val = 255;
          ui->temp_text[1+c*2]=hexdigits[val/16];
          ui->temp_text[1+c*2+1]=hexdigits[val%16];
        }

        if ((widget->flags & UI_NO_ALPHA))
          ui->temp_text[7]=0;
        else
          ui->temp_text[9]=0;

        // TODO : minify string if matching hexdigits


        widget->state = ui_state_commit;
      }

      //ui_choice_draw (ui, x, y);

      ui_end_overlay (ui);
    }

  if (widget->state == ui_state_commit) {
    widget->state = ui_state_default;
    return ctx_strdup(ui->temp_text);
  }
  return NULL;
};

int ui_entry_realloc(Ui *ui, const char *label, const char *fallback,
                     char **strptr) {
  const char *str = "";
  if (*strptr)
    str = *strptr;
  char *ret = ui_entry(ui, label, fallback, str);
  if (ret) {
    if (*strptr)
      ctx_free(*strptr);
    *strptr = ret;
    return 1;
  }
  return 0;
}

char *ui_entry (Ui *ui, const char *label, const char *fallback,
                const char *value)
{
  char *ret = NULL;
  STYLE_TEXT;
  ctx_save(ui->ctx);
  ctx_font_size(ui->ctx, 1.0 * em);
  ctx_move_to(ui->ctx, ui->x0, ui->y + em);
  ctx_text(ui->ctx, label);
  float width = ctx_text_width (ui->ctx, label);

  if ((ret = ui_entry_coords(ui, UI_ID_STR(label), ui->x0 + width + em * 0.5, ui->y,
                             ui->width - width - em, ui->line_height, fallback,
                             value))) {
  }
  ui_y_advance (ui, ui->line_height * 1.4);
  ctx_restore(ui->ctx);

  return ret;
}


char *ui_color_popup (Ui *ui, const char *label, const char *value)
{
  char *ret = NULL;
  ctx_save(ui->ctx);
  ctx_font_size(ui->ctx, 1.0 * em);
  ctx_move_to(ui->ctx, ui->x0, ui->y + em);
  ctx_text(ui->ctx, label);
  float width = ctx_text_width (ui->ctx, label);

  if ((ret = ui_color_popup_coords(ui, UI_ID_STR(label), ui->x0 + width + em * 0.5, ui->y,
                             ui->line_height, ui->line_height,
                             value))) {
  }
  ui_y_advance (ui, ui->line_height * 1.4);
  ctx_restore(ui->ctx);

  return ret;
}


void ui_destroy(Ui *ui) {
  if (!ui)
    return;
  bool owns_ctx = ui->owns_ctx;
  Ctx *ctx = ui->ctx;
  // XXX : destroy more
  free(ui);
  if (owns_ctx)
    ctx_destroy(ctx);
}

Ctx *ui_ctx(Ui *ui) { return ui->ctx; }

float ui_x(Ui *ui) {
  return 0.0f; // return ui->x;
}

float ui_y(Ui *ui) { return ui->y; }

void ui_move_to(Ui *ui, float x, float y) {
  ui->x = x;
  ui->y = y;
}

#define DEF_SLIDER(type)                                                       \
  void ui_slider_##type(Ui *ui, const char *label, type *val, type min,        \
                        type max, type step) {                                 \
    *val = ui_slider(ui, label, min, max, step, *val);                         \
  }

DEF_SLIDER(float)
DEF_SLIDER(int)
DEF_SLIDER(uint8_t)
DEF_SLIDER(uint16_t)
DEF_SLIDER(uint32_t)
DEF_SLIDER(int8_t)
DEF_SLIDER(int16_t)
DEF_SLIDER(int32_t)
#undef DEF_SLIDER

void ui_add_key_binding(Ui *ui, const char *key, const char *action,
                        const char *label)
{
  ctx_add_key_binding(ui->ctx, key, action, label, ui_cb_do, ui);
}

void *ui_last_id (Ui *ui)
{
  return ui->last_id;
}

void *ui_focused_id (Ui *ui)
{
  return ui->focused_id;
}

void ui_ignore_cb (CtxEvent *e, void *a, void *b)
{
  Ui *ui = a;
  if (e->type == CTX_DRAG_RELEASE)
  {
    ui_do (ui, "close-overlay");
  }
  e->stop_propagate = 1;
}

void ui_start_overlay (Ui *ui, float fade_bg)
{
  ui->overlay[ui->overlay_no].x = ui->x;
  ui->overlay[ui->overlay_no].y = ui->y;
  ui->overlay[ui->overlay_no].x0 = ui->x0;
  ui->overlay[ui->overlay_no].y0 = ui->y0;
  ui->overlay[ui->overlay_no].width = ui->width;
  ui->overlay[ui->overlay_no].height = ui->height;

  ui->overlay_no ++;
  if (ui->overlay_no > ui->max_overlay)
    ui->max_overlay = ui->overlay_no;

  if (ui->prev_max_overlay < ui->max_overlay)
  {
    ui->overlay[ui->max_overlay-1].focused_id = ui->focused_id;
  }

  ctx_get_transform (ui->ctx, 
      &ui->overlay[ui->overlay_no].a,
      &ui->overlay[ui->overlay_no].b,
      &ui->overlay[ui->overlay_no].c,
      &ui->overlay[ui->overlay_no].d,
      &ui->overlay[ui->overlay_no].e,
      &ui->overlay[ui->overlay_no].f,
      &ui->overlay[ui->overlay_no].g,
      &ui->overlay[ui->overlay_no].h,
      &ui->overlay[ui->overlay_no].i);

  ui->overlay[ui->overlay_no].drawlist = ctx_new_drawlist (ctx_width (ui->ectx), ctx_height (ui->ectx));
  ui->ctx = ui->overlay[ui->overlay_no].drawlist;
  ctx_save (ui->ctx);

  ctx_rectangle (ui->ectx, -ctx_width(ui->ctx), -ctx_height (ui->ctx), ctx_width (ui->ctx) * 2, ctx_height (ui->ctx) * 2);
  ctx_listen (ui->ectx, CTX_DRAG|CTX_SCROLL|CTX_MOTION, ui_ignore_cb, ui, NULL);
  ctx_reset_path (ui->ectx);

  if (fade_bg < 1.0)
  {
    ctx_rectangle (ui->ctx, -ctx_width(ui->ctx), -ctx_height (ui->ctx), ctx_width (ui->ctx) * 2, ctx_height (ui->ctx) * 2);
    ui_color_base_alpha (ui, 0.5, 0.0, 1.0-fade_bg);
    ctx_fill (ui->ctx);
  }

  ctx_apply_transform (ui->ctx,
         ui->overlay[ui->overlay_no].a,
         ui->overlay[ui->overlay_no].b,
         ui->overlay[ui->overlay_no].c,
         ui->overlay[ui->overlay_no].d,
         ui->overlay[ui->overlay_no].e,
         ui->overlay[ui->overlay_no].f,
         ui->overlay[ui->overlay_no].g,
         ui->overlay[ui->overlay_no].h,
         ui->overlay[ui->overlay_no].i);
}
void ui_end_overlay (Ui *ui)
{
  ctx_restore (ui->ctx);
  ui->overlay_no --;
  ui->x = ui->overlay[ui->overlay_no].x;
  ui->y = ui->overlay[ui->overlay_no].y;
  ui->x0 = ui->overlay[ui->overlay_no].x0;
  ui->y0 = ui->overlay[ui->overlay_no].y0;
  ui->width = ui->overlay[ui->overlay_no].width;
  ui->height = ui->overlay[ui->overlay_no].height;

  if (ui->overlay_no)
  {
    ui->ctx = ui->overlay[ui->overlay_no].drawlist;
  }
  else
  {
    ui->ctx = ui->ectx;
  }
}

int ui_choice_coords (Ui *ui, float x, float y, float width, float height,
                      const char *label, void *id, int value)
{
  UiFlag flags = ui->flags;
  Ctx *ctx = ui->ctx;
  Ctx *ectx = ui->ectx;
  if (width <= 0)
    width = ctx_text_width(ctx, label);
  if (height <= 0)
    height = em * 1.4;

  UiWidget *widget =
      ui_widget_register(ui, ui_type_choice, x, y, width, height, id);

  ctx_save (ctx);
    

  if (! (flags & UI_NO_LABEL))
  {
    STYLE_TEXT;
    ctx_move_to (ctx, x, y + em);
    ctx_text (ctx, label);
    ctx_text (ctx, ":");

  ui->choice_x = x + ctx_text_width (ctx, label) + ctx_text_width (ctx, ":") + em;
  }
  else
  {
    ui->choice_x = x;
  }
 
  ui->choice_name = ctx_strdup (label);

  if (widget->fresh)
    widget->float_data = value;

  if (ui->overlay_no == ui->max_overlay)
  {
    ctx_reset_path(ectx);
    ctx_rectangle(ectx, x, y, width, height);
    ctx_listen(ectx, CTX_DRAG, ui_choice_drag, ui, widget->id);
    ctx_listen_set_cursor (ectx, CTX_CURSOR_HAND);
    ctx_reset_path(ectx);
  }


  if (flags & UI_INLINE)
  {
    ui_choice_draw (ui, x, y);
  }
  else
  {
    ui_choice_flush (ui);
  }
  ui_choice_clear (ui);
  ctx_restore (ctx);
  return widget->float_data;
}

int ui_choice (Ui *ui, const char *label, int value)
{
  UiFlag flags = ui->flags;
  if (flags & UI_INLINE)
  {
    float width = ctx_text_width (ui->ctx, label);
    width += em;
    ui->x += width + em * 0.5;
    if (ui->x > ui->x0 + ui->width)
    {
      ui->x = ui->x0 + width + em * 0.5;
      ui->y += ui->line_height;
    }
  
    return ui_choice_coords (ui, ui->x - width - em * 0.5, ui->y,
                             width, ui->line_height * 0.8, label,
                             UI_ID_STR(label), value);
  }
  ui_y_advance (ui, ui->line_height);
  return ui_choice_coords (ui, ui->x, ui->y - ui->line_height,
                           ui->width, ui->line_height * 0.8, label,
                           UI_ID_STR(label), value);
}

void ui_choice_add (Ui *ui, const char *label, int value)
{
  if (ui->n_choices + 1 >= UI_MAX_CHOICES)
  {
    fprintf (stderr, "too many choices\n");
    return;
  }
  ui->choice_names[ui->n_choices] = ctx_strdup (label);
  ui->choice_values[ui->n_choices] = value;
  ui->n_choices ++;
}

void ui_flags (Ui *ui, UiFlag flags)
{
  ui->flags = flags;
}

int ui_main (Ui *ui_in, void (*fun)(Ui *ui_in, float delta_s, void *data), void *data)
{
  Ui *ui = ui_in;
  Ctx *ctx = NULL;
  if (!ui)
  {
    ctx = ctx_new(-1, -1, NULL);
    if (!ctx)
      return -1;
    ui = ui_for_ctx (ctx);
    if (!ui)
      return -1;
  }

  while (!ctx_has_exited(ctx))
  {
    if (ctx_need_redraw (ctx))
    {
      float delta_s = ctx_start_frame(ctx);
      if (delta_s > 0.2)
        delta_s = 0.0f;
      ui_start_frame (ui);
      fun (ui, delta_s, data);
      ui_end_frame (ui);
      ctx_end_frame(ctx);
    }
    else
    {
      ctx_handle_events (ctx);
    }
  }
  if (ui != ui_in)
  {
    ui_destroy (ui);
    ctx_destroy(ctx);
  }
  return 0;
}

#undef em

