/* a pong style game
 *
 * showing how to register for key-up / key-down events
 * the same game should also be fully operable with touch input
 */


#include <math.h>
#include "ctx.h"

typedef struct Shape
{
  float x, y;
  float width, height;

  int   n_points;
  float points[32][2];
} Shape;


typedef struct Player
{
  Shape *shape;
  int moving_up;
  int moving_down;
  int score;
} Player;


typedef struct Playfield
{
  Shape  shape[16];

  int    n_shapes;
  float  width;
  float  height;

  Player player[2];

  float  ball_xd;
  float  ball_yd;

  float  player_speed;

  int ai;

  float  cur_time;
} Playfield;
#define POLYLINE  0
#define CIRCLE    1
#define RECTANGLE 2

void shape_add_point (Shape *shape, float x, float y)
{
  int no = shape->n_points++;
  shape->points[no][0]=x;
  shape->points[no][1]=y;
}

Shape *playfield_add_shape (Playfield *p, float x, float y, float width, float height, int type)
{
  int no = p->n_shapes++;
  p->shape[no].x = x;
  p->shape[no].y = y;
  p->shape[no].width = width;
  p->shape[no].height = height;
  p->shape[no].n_points = type;
  if (type == RECTANGLE)
  {
    p->shape[no].n_points = POLYLINE;
    shape_add_point (&p->shape[no], 0, 0);
    shape_add_point (&p->shape[no], width, 0);
    shape_add_point (&p->shape[no], width, height);
    shape_add_point (&p->shape[no], 0, height);
  }
  return &p->shape[no];
}

void player_reshape (Playfield *p, int player_no, float height)
{
  Shape *s = p->player[player_no].shape;
  
  s->width = 2;
  s->height = height;

  float dir = player_no?1:-1;

  s->n_points = POLYLINE;

  float b = 1.0;
  if (player_no) b = 1.0;

  shape_add_point (s, b - dir * 0.5, 0);
  shape_add_point (s, b - dir * 0.5, height);

  shape_add_point (s, b + dir * 0.05, height * 1.00);
  shape_add_point (s, b + dir * 0.23, height * 0.80);
  shape_add_point (s, b + dir * 0.25, height * 0.75);
  shape_add_point (s, b + dir * 0.29, height * 0.55);
  shape_add_point (s, b + dir * 0.29, height * 0.45);
  shape_add_point (s, b + dir * 0.25, height * 0.25);
  shape_add_point (s, b + dir * 0.23, height * 0.20);
  shape_add_point (s, b + dir * 0.05, height * 0.00);
}


Playfield *playfield_new (void)
{
  Playfield *p = calloc (1, sizeof (Playfield));
  p->width  = 40;
  p->height = 30;

  playfield_add_shape (p, p->width/2, p->height/2, 1, 1, CIRCLE);

  p->player[0].shape = playfield_add_shape (p, p->width-2, 3, 1, p->height/4, RECTANGLE);
  p->player[1].shape = playfield_add_shape (p, 0, 3, 1, p->height/4, RECTANGLE);

  playfield_add_shape (p, 0, p->height-0.3, p->width, 0.3, RECTANGLE);
  playfield_add_shape (p, 0, 0, p->width, 0.3, RECTANGLE);

  p->ball_xd = 8;
  p->ball_yd = 2;
  p->player_speed = 15;

  player_reshape (p, 0, 5);
  player_reshape (p, 1, 5);

  return p;
}

void playfield_destroy (Playfield *p)
{
  free (p);
}
float seconds_in_state = 0.0f;

typedef enum GameState {
  STATE_INIT         = -1,
  STATE_WELCOME      = 0,
  STATE_P1_ACTIVE    = 1,
  STATE_P2_ACTIVE    = 2,
  STATE_GAME_RUNNING = 3,
  STATE_GAME_WON     = 8,
} GameState;

GameState state = STATE_INIT;

void player_drag (CtxEvent *e, void *a, void *b)
{
  Playfield *p = a;
  Player *player = b;

  player->shape->y += e->delta_y;
  if (p->ai == 3)
  {
    p->ai = 0;
    state = 0;
    p->player[0].moving_down = 0;
    p->player[0].moving_up = 0;
    p->player[1].moving_down = 0;
    p->player[1].moving_up = 0;
    p->shape[0].x = p->width/2;
    p->shape[0].y = p->height/2;
  }
}

void playfield_scale_to_fit (Playfield *p, Ctx *ctx)
{
  float scale = ctx_width (ctx) / p->width;
  if (ctx_height (ctx) / p->height < scale)
    scale = ctx_height (ctx) / p->height;

  ctx_translate (ctx, 
      (ctx_width(ctx) - p->width*scale)/2,
      (ctx_height(ctx) - p->height*scale)/2);

  ctx_scale (ctx, scale, scale);
}


float in_same_state = 0.0f;


void playfield_ai (Playfield *p)
{
  for (int player_no = 0; player_no < 2; player_no++)
  {
    Player *player = &p->player[player_no];
    if (p->ai & (1 << player_no))
    {
      Shape *shape = player->shape;

      if (player_no == 0 && p->ball_xd < 0)
      {
        player->moving_down = 0;
        player->moving_up = 0;
      }
      else if (player_no == 1 && p->ball_xd > 0)
      {
        player->moving_down = 0;
        player->moving_up = 0;
      }
      else if (shape->y + shape->height/2 > p->shape[0].y + 1.8)
      {
        player->moving_up = 1;
        player->moving_down = 0;
      }
      else if (shape->y + shape->height/2 < p->shape[0].y - 1.8)
      {
        player->moving_up = 0;
        player->moving_down = 1;
      }
      else
      {
        player->moving_down = 0;
        player->moving_up = 0;
      }
    }

    if (player_no == 0 && (state == STATE_WELCOME) && (in_same_state > 5))
    {
      player->moving_down = 1;
      p->ai |= (1 << player_no);
      break;
    }
    if (player_no == 1 && (state == STATE_P1_ACTIVE) && (in_same_state > 3))
    {
      player->moving_down = 1;
      p->ai |= (1 << player_no);
      break;
    }
    if (player_no == 0 && (state == STATE_P2_ACTIVE) && (in_same_state > 3))
    {
      player->moving_down = 1;
      p->ai |= (1 << player_no);
      break;
    }
  }

}


void playfield_draw (Playfield *p, void *ctx, float delta_s)
{
  ctx_save (ctx);

  playfield_scale_to_fit (p, ctx);


  ctx_rectangle (ctx, 0, 0, p->width, p->height);
  ctx_rgba (ctx, 0,0, 0, 1);
  ctx_fill (ctx);


  ctx_rgba (ctx, 0.5,0.5, 0.5, 1);

  char buf[128];

  static GameState old_state = 0;

  if (old_state == state)
    in_same_state += delta_s;
  else
    in_same_state = 0;

  if (state == STATE_INIT)
  {
    p->ai = 0;
    sprintf (buf, "--");
    p->player[0].score = 0;
    p->player[1].score = 0;
    p->player[0].shape->y = p->height/2 - p->player[0].shape->height/2;
    p->player[1].shape->y = p->height/2 - p->player[1].shape->height/2;
    state = STATE_WELCOME;
  }
  else if (state == STATE_WELCOME)
  {
    sprintf (buf, "wiggle up-down or pad to enter game\n");
    if (p->player[0].moving_up | p->player[0].moving_down |
       (p->player[0].shape->y != p->height/2 - p->player[0].shape->height/2))
    {
      state |= STATE_P1_ACTIVE;
      in_same_state = 0;
    }
    if (p->player[1].moving_up| p->player[1].moving_down |
      (p->player[1].shape->y != p->height/2 - p->player[1].shape->height/2))
    {
      state |= STATE_P2_ACTIVE;
      in_same_state = 0;
    }
  }
  else if (state == STATE_P1_ACTIVE)
  {
    sprintf (buf, "wiggle a-z / left pad to enter game\n");
    if (p->player[1].moving_up| p->player[1].moving_down |
      (p->player[1].shape->y != p->height/2 - p->player[1].shape->height/2))
    {
      state |= STATE_P2_ACTIVE;
      in_same_state = 0;
    }
  }
  else if (state == STATE_P2_ACTIVE)
  {
    sprintf (buf, "wiggle up-down / right pad to enter game\n");
    if (p->player[0].moving_up | p->player[0].moving_down |
       (p->player[0].shape->y != p->height/2 - p->player[0].shape->height/2))
    {
      state |= STATE_P1_ACTIVE;
      in_same_state = 0;
    }
  }
  else if (state == STATE_GAME_RUNNING)
  {
    sprintf (buf, "%i - %i", p->player[1].score, p->player[0].score);
  }
  else if (state & STATE_GAME_WON)
  {
    if (p->player[1].score > p->player[0].score)
      sprintf (buf, "<-- winner");
    else
      sprintf (buf, "winner -->");
    if (in_same_state > 7)
    {
      state = STATE_INIT;
      in_same_state = 0;
    }
  }

  old_state = state;

  float font_size = p->height / 15;
  ctx_font_size (ctx, font_size);
  ctx_move_to (ctx, p->width/2, font_size * 1.2);
  ctx_text_align (ctx, CTX_TEXT_ALIGN_CENTER);
  ctx_text (ctx, buf);

  for (int i = 0 ; i < p->n_shapes; i++)
  {
    Shape *shape = &p->shape[i];
    switch (shape->n_points)
    {
      case 0: // unused
        break;
      case 1: // circle
        ctx_arc (ctx, shape->x + shape->width/2, shape->y+shape->width/2, shape->width/2, 0.0, 3.1415*2, 0);
        break;
      case 2: // rectangle
        ctx_rectangle (ctx, shape->x, shape->y, shape->width, shape->height);
        break;
      default:
        ctx_save (ctx);
        ctx_translate (ctx, shape->x, shape->y);
        for (int i = 0; i < shape->n_points; i++)
        {
          if (i == 0)
            ctx_move_to (ctx, shape->points[i][0],
                              shape->points[i][1]);
          else
            ctx_line_to (ctx, shape->points[i][0],
                              shape->points[i][1]);
        }
        ctx_restore (ctx);
        break;
    }
    ctx_rgba (ctx, 1, 1, 1, 1);
    ctx_fill (ctx);
  }
  ctx_line_width (ctx, 0.2);

  ctx_reset_path (ctx);
  for (int i = 0 ; i < 2; i++)
  {
    Player *player = &p->player[i];
    Shape *shape = player->shape;
    ctx_reset_path (ctx);
    ctx_rectangle (ctx, shape->x, shape->y, shape->width, shape->height);

    ctx_listen (ctx, CTX_DRAG, player_drag, p, player);
    ctx_rgba (ctx, 0,1,1,0.5);
    //ctx_stroke (ctx);
    ctx_reset_path (ctx);
  }

  ctx_restore (ctx);
}

void playfield_time_step (Playfield *p, float seconds_elapsed)
{
  Shape *ball = &p->shape[0];

  for (int i = 0; i < 2; i++)
  {
    Player *player = &p->player[i];

    // update position at constant max speed
    if (player->moving_down)
      player->shape->y += seconds_elapsed * p->player_speed;
    if (player->moving_up)
      player->shape->y -= seconds_elapsed * p->player_speed;

    // constrain vertical position
    //
    if (player->shape->y < 0)
      player->shape->y = 0;
    if (player->shape->y + player->shape->height > p->height)
      player->shape->y = p->height - player->shape->height;
  }


  if (state != STATE_GAME_RUNNING)
    return;



  float x = ball->x + 0.5;
  float y = ball->y + 0.5;

  float nx = x + seconds_elapsed * p->ball_xd;
  float ny = y + seconds_elapsed * p->ball_yd;

  float intersect_x0;
  float intersect_y0;
  float intersect_x1;
  float intersect_y1;

  int intersects = 0;
  for (int i = 1; !intersects && i < p->n_shapes; i++)
  {
    Shape *obstacle = &p->shape[i];

    for (int edge = 0; !intersects && edge < obstacle->n_points; edge++)
    {
      float x1 = obstacle->points[edge][0] + obstacle->x;
      float y1 = obstacle->points[edge][1] + obstacle->y;

      float x2 = obstacle->points[(obstacle->n_points-1 == edge)?0:(edge+1)][0] + obstacle->x;
      float y2 = obstacle->points[(obstacle->n_points-1 == edge)?0:(edge+1)][1] + obstacle->y;

      float distance =
             fabsf ((y2-y1)*nx - (x2-x1)*ny + x2*y1 - y2 * x1) /
             sqrtf ( (y2 - y1)*(y2-y1) + (x2-x1) * (x2-x1));

      float minx = x1; if (x2<minx) minx = x2;
      float maxx = x1; if (x2>maxx) maxx = x2;

      float miny = y1; if (y2<miny) miny = y2;
      float maxy = y1; if (y2>maxy) maxy = y2;

      float radius = 0.5f;

      if ((distance < radius) &&
          (nx >= minx - radius) && (nx <= maxx + radius) && 
          (ny >= miny - radius) && (ny <= maxy + radius)
          )
      {
        intersect_x0 = x1;
        intersect_y0 = y1;
        intersect_x1 = x2;
        intersect_y1 = y2;
        float angle = atan2f(intersect_x1-intersect_x0,
                             intersect_y1-intersect_y0);

        float tx = p->ball_xd * cosf(angle) - p->ball_yd * sinf (angle);
        float ty = p->ball_xd * sinf(angle) + p->ball_yd * cosf (angle);

        tx = -tx;

        p->ball_xd = tx * cosf(-angle) - ty * sinf (-angle);
        p->ball_yd = tx * sinf(-angle) + ty * cosf (-angle);

        nx = nx + seconds_elapsed * p->ball_xd * 2;
        ny = ny + seconds_elapsed * p->ball_yd * 2;

        float vmag = hypotf (p->ball_xd, p->ball_yd);
        p->ball_xd = p->ball_xd/vmag * (vmag + 2.0f);
        p->ball_yd = p->ball_yd/vmag * (vmag + 2.0f);
    }
  }
  }

  x = nx;
  y = ny;

  ball->x = x - 0.5;
  ball->y = y - 0.5;

  if (ball->x < -1)
  {
    p->player[0].score ++;
    ball->x = p->width/2;
    ball->y = p->height/2;
    if (p->player[0].score > 4)
      state = STATE_GAME_WON;
    //p->ball_xd *= -1;
    p->ball_xd = 8;
    p->ball_yd = (((random()%255)/255.0)-0.5) * 8;
  }
  if (ball->x > p->width)
  {
    p->player[1].score ++;
    if (p->player[1].score > 4)
      state = STATE_GAME_WON;
    ball->x = p->width/2;
    ball->y = p->height/2;
    p->ball_xd = -8;
    p->ball_yd = (((random()%255)/255.0)-0.5) * 8;
  }
}

void handle_keys (CtxEvent *e, void *a, void *b)
{
  Playfield *p = a;
  int pressed = e->type == CTX_KEY_DOWN;
  if (!strcmp (e->string, "up"))
    p->player[0].moving_up = pressed;
  else if (!strcmp (e->string, "down"))
    p->player[0].moving_down = pressed;
  else if (!strcmp (e->string, "a"))
    p->player[1].moving_up = pressed;
  else if (!strcmp (e->string, "z"))
    p->player[1].moving_down = pressed;

  if (p->ai == 3)
  {
    p->ai = 0;
    state = 0;
    p->player[0].moving_down = 0;
    p->player[0].moving_up = 0;
    p->player[1].moving_down = 0;
    p->player[1].moving_up = 0;
    p->shape[0].x = p->width/2;
    p->shape[0].y = p->height/2;
  }
}

void exit_cb (CtxEvent *e, void *a, void *b)
{
  ctx_exit (e->ctx);
}

void playfield_loop (Ctx *ctx, float delta_s, void *user_data)
{
  Playfield *p = user_data;

  playfield_draw (p, ctx, delta_s);
  playfield_ai (p);

  ctx_add_key_binding(ctx, "escape", "exit", "quit game", exit_cb, NULL);
  ctx_reset_path (ctx);
  ctx_listen (ctx, CTX_KEY_UP|CTX_KEY_DOWN, handle_keys, p, NULL);

  float target_time = p->cur_time + delta_s;
  float time_step = 1/100.0f;
  while (p->cur_time < target_time)
  {
    playfield_time_step (p, time_step);
    p->cur_time += time_step;
  }
  ctx_queue_draw (ctx);
}

#if ESP_PLATFORM
void app_main(void)
#else
int main(int argc, char **argv) 
#endif
{
  Playfield *p = playfield_new ();
  ctx_main (NULL, playfield_loop, p);
  playfield_destroy (p);
#ifndef ESP_PLATFORM
  return 0;
#endif
}
