#include "ctx_config.h"

#if CTX_VT

#include <unistd.h>
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <math.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <time.h>

#include "ctx.h"

void
vt_send_message (VT *vt, const char *message);

#include "ui/ui.h"
#include "ui/ui.c"
#undef em

#define LAYOUT_MAXIMIZED 0
#define LAYOUT_WINDOWS   1
#define LAYOUT_FOUR_BY_THREE 2


#define STYLE_DARK_BRIGHTNESS  (15.0f)
#define STYLE_DARK_CONTRAST    (1.0f)
#define STYLE_DARK_SATURATION  (170.0f)

#define STYLE_LIGHT_BRIGHTNESS  (90.0f)
#define STYLE_LIGHT_CONTRAST    (1.0f)
#define STYLE_LIGHT_SATURATION  (100.0f)

#define STYLE_LIGHT_HC_BRIGHTNESS  (100.0f)
#define STYLE_LIGHT_HC_CONTRAST    (5.0f)
#define STYLE_LIGHT_HC_SATURATION  (130.0f)

#define STYLE_DARK_HC_BRIGHTNESS  (0.0f)
#define STYLE_DARK_HC_CONTRAST    (5.0f)
#define STYLE_DARK_HC_SATURATION  (130.0f)

#define STYLE_EINK_BRIGHTNESS  (100.0f)
#define STYLE_EINK_CONTRAST    (5.0f)
#define STYLE_EINK_SATURATION  (0.0f)

extern CtxSubPixel _ctx_subpixel_layout;// = CTX_SUBPIXEL_HRGB;
static Ui *ui = NULL;

CtxClient *second = NULL; // XXX at least give better name
static float    animation_duration = 0.175f;//25f;
static Ctx     *ctx = NULL; // initialized in main
static char    *ctx_config = NULL;
static uint64_t ctx_config_timestamp = 0;
static float    ctx_cfg_dirty = 0.0;
static float    help_elapsed = 0.0f;

static float backlight = 100.0f;

int magnifier_active = 0;
static float full_time = 0.0f;
int panel_open = 0;
int panel_focused = 0;

//static int cfg_has_reloaded = 0;
static int layout_no = LAYOUT_FOUR_BY_THREE;

#define ENABLE_PORTRAIT 1

#include "cfg.h"

static int locked = 0;

void ctx_set_focus_cb (Ctx *ctx, void(*focus_cb)(Ctx *ctx, int id, void *user_data), void *user_data);

typedef struct _CtxClient CtxClient;
CtxList *ctx_clients(Ctx *ctx);

int ctx_client_get_width (Ctx *ctx, int id);
int ctx_client_get_height (Ctx *ctx, int id);
int ctx_client_get_x (Ctx *ctx, int id);
int ctx_client_get_y (Ctx *ctx, int id);

void ctx_screenshot (Ctx *ctx, const char *path);
void
vt_screenshot (const char *output_path)
{
  ctx_screenshot (ctx, output_path);
}

void ctx_client_lock (CtxClient *client);
void ctx_client_unlock (CtxClient *client);

#define VT_RECORD 0
static int terminal_no_new_tab = 0;

static char *execute_self = NULL;

static float font_size    = 19.7f;
#define em font_size
static float titlebar_height = 1.5;
static float line_spacing = 1.0f;
static float baseline     = 0.78f;


float add_x = 10;
float add_y = 100;

#if ENABLE_PORTRAIT
  // move this code to ui?
static int orientation = 0;

static uint64_t o_start = 0;

#define TERMINAL_BATTERY_STATUS 1
#if TERMINAL_BATTERY_STATUS
    #include <stdio.h>
    #include <string.h>
    #include <dirent.h>
    #include <stdlib.h>

float battery_status_linux(int *is_charging)
{
  float battery_level = 100.0f;
    DIR *dir;
    struct dirent *entry;
    char path[512];
    char buffer[128];
    FILE *fp;

    dir = opendir("/sys/class/power_supply/");
    if (dir == NULL) {
        perror("Error opening /sys/class/power_supply/");
        return 1;
    }

    while ((entry = readdir(dir)) != NULL)
    {
      snprintf(path, sizeof(path), "/sys/class/power_supply/%s/capacity", entry->d_name);
      fp = fopen(path, "r");
      if (fp)
      {
        if (fgets(buffer, sizeof(buffer), fp) != NULL)
        {
          battery_level = atoi (buffer);
        }
        fclose(fp);

        snprintf(path, sizeof(path), "/sys/class/power_supply/%s/status", entry->d_name);
        fp = fopen(path, "r");
        if (is_charging && fp)
        {
          if (fgets(buffer, sizeof(buffer), fp) != NULL) {
            if (strstr (buffer, "Charging"))
              {
                *is_charging = 1;
              }
              else
                *is_charging = 0;
              }
              fclose(fp);
        }
      }
    }
    closedir(dir);
    return battery_level;
}

float battery_status (int *is_charging)
{
  return battery_status_linux(is_charging);
}
#endif

static inline int is_portrait (Ctx *ctx)
{
  return (orientation != 0);
}

float ctx_virtual_rotation (Ctx *ctx)
{
  float a = 0.0f;
  float b = M_PI/2;

  float t = (ctx_ticks()-o_start) / 1000.0f / 1000.0f / animation_duration;
  if (t < 1.0f && o_start)
  {
    ctx_queue_draw (ctx);
    if (is_portrait (ctx))
      return a * (1.0-t) + t * b;
    return b * (1.0-t) + t * a;
  }

  if (is_portrait(ctx))
    return b;
  return a;
}

float ctx_virtual_width (Ctx *ctx)
{
  float a = ctx_width (ctx);
  float b = ctx_height (ctx);

  float t = (ctx_ticks()-o_start) / 1000.0f / 1000.0f / animation_duration;
  if (t < 1.0f && o_start)
  {
    //ctx_queue_draw (ctx);
    if (is_portrait (ctx))
      return a * (1.0-t) + t * b;
    return b * (1.0-t) + t * a;
  }

  if (is_portrait(ctx))
    return b;
  return a;
}

float ctx_virtual_height (Ctx *ctx)
{
  float a = ctx_height (ctx);
  float b = ctx_width (ctx);


  float t = (ctx_ticks()-o_start) / 1000.0f / 1000.0f / animation_duration;
  if (t < 1.0f && o_start)
  {
    ctx_queue_draw (ctx);
    if (is_portrait (ctx))
      return a * (1.0-t) + t * b;
    return b * (1.0-t) + t * a;
  }

  if (is_portrait(ctx))
    return b;
  return a;
}

const char *get_icon (const char *label);
static void icon (Ctx *ctx, float x, float y, float w, float h, const char *name);

#include "terminal-keyboard.h"

static void rotate (Ctx *ctx)
{
  orientation++;
  if (orientation > 1)
    orientation = 0;
  o_start = ctx_ticks ();
  ctx_queue_draw (ctx);
}

static void rotate_cb (CtxEvent *event, void *a, void *b)
{
  Ctx *ctx = event->ctx;
  rotate (ctx);
}
#endif
/********************/

static char *ctx_term_config_dir (void)
{
  static char *res = NULL; // will be leaked
  if (res) return res;
  if (getenv ("XDG_CONFIG_HOME"))
    res = ctx_strdup_printf ("%s/ctx", getenv("XDG_CONFIG_HOME"));
  else
    res = ctx_strdup_printf ("%s/.config/ctx", getenv("HOME"));
  return res;
}

static char *ctx_term_config_path (void)
{
  static char *res = NULL; // will be leaked
  if (res) return res;
    res = ctx_strdup_printf ("%s/ctx.ini", ctx_term_config_dir ());
  return res;
}

static char *ctx_term_lock_path (void)
{
  static char *res = NULL; // will be leaked
  if (res) return res;
    res = ctx_strdup_printf ("%s/ctx.lock", ctx_term_config_dir ());
  return res;
}

static const char *ctx_find_shell_command (void)
{
  if (access ("/.flatpak-info", F_OK) != -1)
  {
    static char ret[512];
    char buf[256];
    FILE *fp = popen("flatpak-spawn --host getent passwd $USER|cut -f 7 -d :", "r");
    if (fp)
    {
      while (fgets (buf, sizeof(buf), fp) != NULL)
      {
        if (buf[strlen(buf)-1]=='\n')
          buf[strlen(buf)-1]=0;
        sprintf (ret, "flatpak-spawn --env=TERM=xterm --host %s", buf);
      }
      pclose (fp);
      return ret;
    }
  }

  if (getenv ("SHELL"))
  {
    return getenv ("SHELL");
  }
  int i;
  const char *command = NULL;
  struct stat stat_buf;
  static const char *alts[][2] =
  {
    {"/bin/bash",     "/bin/bash"},
    {"/usr/bin/bash", "/usr/bin/bash"},
    {"/bin/sh",       "/bin/sh"},
    {"/usr/bin/sh",   "/usr/bin/sh"},
    {NULL, NULL}
  };
  for (i = 0; alts[i][0] && !command; i++)
    {
      lstat (alts[i][0], &stat_buf);
      if (S_ISREG (stat_buf.st_mode) || S_ISLNK (stat_buf.st_mode) )
        { command = alts[i][1]; }
    }
  return command;
}

/*****************/
#define flag_is_set(a, f) (((a) & (f))!=0)
#define flag_set(a, f)    ((a) |= (f));
#define flag_unset(a, f)  ((a) &= ~(f));

void update_vts_light_mode (Ctx *ctx)
{
  CtxList *l = ctx_clients (ctx);
  for (; l; l = l->next)
  {
    CtxClient *client = l->data;
    VT *vt = ctx_client_vt (client);

    if (vt)
    {
      float rgba_base[4];
      float rgba_ink[4];

      float bg_temp = cfg_f32 (&ctx_config, "style", "background-temperature");
      float brightness = cfg_f32 (&ctx_config, "style", "brightness");
      if (brightness > 70)
      {
        // TODO : make smooth transition
        ui_color_compute (ui, UI_COLOR_BASE, 1.0, bg_temp, 1, rgba_base);
      }
      else
      {
        ui_color_compute (ui, UI_COLOR_BASE, 0.5, bg_temp, 1, rgba_base);
      }
      ui_color_compute (ui, UI_COLOR_INK, 1.0, 0, 1.0, rgba_ink);
      vt_set_palette (vt, -2, rgba_base[0]*255, rgba_base[1]*255, rgba_base[2]*255);
      vt_set_palette (vt, -1, rgba_ink[0]*255, rgba_ink[1]*255, rgba_ink[2]*255);
    }
  }
  ctx_queue_draw (ctx);
}

static void update_maximized_size (Ctx *ctx);

void tab_update_theme (VT *vt)
{
  char buf[256];

  // TODO : font-size?

  sprintf (buf, "ui-brightness:%.0f", cfg_f32 (&ctx_config, "style", "brightness"));
  vt_send_message (vt, buf);
  sprintf (buf, "ui-saturation:%.0f", cfg_f32 (&ctx_config, "style", "saturation"));
  vt_send_message (vt, buf);

  sprintf (buf, "ui-base-saturation:%.0f", cfg_f32 (&ctx_config, "style", "base-saturation"));
  vt_send_message (vt, buf);

  sprintf (buf, "ui-contrast:%.2f", cfg_f32 (&ctx_config, "style", "contrast"));
  vt_send_message (vt, buf);
  
  char *str = cfg_string_full (&ctx_config, "style", "brightest", "", NULL);
  if (str)
  {
    sprintf (buf, "ui-brightest:%s", str);
    vt_send_message (vt, buf);
    free (str);
  }

  str = cfg_string_full (&ctx_config, "style", "darkest", "", NULL);
  if (str)
  {
    sprintf (buf, "ui-darkest:%s", str);
    vt_send_message (vt, buf);
    free (str);
  }

  str = cfg_string_full (&ctx_config, "style", "warm", "", NULL);
  if (str)
  {
    sprintf (buf, "ui-warm:%s", str);
    vt_send_message (vt, buf);
    free (str);
  }
  str = cfg_string_full (&ctx_config, "style", "cool", "", NULL);
  if (str)
  {
    sprintf (buf, "ui-cool:%s", str);
    vt_send_message (vt, buf);
    free (str);
  }
}

void tabs_update_theme (Ui *ui)
{
  Ctx *ctx = ui_ctx (ui);
  for (CtxList *l = ctx_clients (ctx); l; l = l->next)
  {
    CtxClient *client = l->data;
    VT *vt = ctx_client_vt (client);
    tab_update_theme (vt);
  }
}

static int idle_update_theme (Ctx *ctx, void *userdata)
{
  tab_update_theme ((VT*)userdata);
  return 0;
}

void terminal_ctx_events_init (VT *vt, void *userdata)
{
  ctx_add_timeout (ctx, 20, idle_update_theme, vt);
}

int add_tab (Ui *ui, Ctx  *ctx, const char *commandline, int can_launch)
{
  float titlebar_h = ctx_height (ctx)/40;
  //int was_maximized = ctx_client_is_maximized (ctx, ctx_clients_active (ctx));
  int flags = CTX_CLIENT_UI_RESIZABLE |  CTX_CLIENT_TITLEBAR;
  if (can_launch) flags |= CTX_CLIENT_CAN_LAUNCH;

  CtxClient *active = ctx_client_new (ctx, commandline, add_x, add_y,
    ctx_virtual_width(ctx)/2, (ctx_virtual_height (ctx) - titlebar_h)/2,
    font_size,
    flags, NULL, NULL);

  VT *vt = ctx_client_vt (active);
  vt_set_line_spacing (vt, line_spacing);
  vt_set_baseline (vt, baseline);
  add_y += ctx_height (ctx) / 20;
  add_x += ctx_height (ctx) / 20;

  if (add_y + ctx_height(ctx)/2 > ctx_client_max_y_pos (ctx))
  {
    add_y = ctx_client_min_y_pos (ctx);
    add_x -= ctx_height (ctx) / 40 * 4;
  }
  ctx_queue_draw (ctx);
  ctx_client_focus (ctx, ctx_client_id (active));

  switch (layout_no)
  {
    case LAYOUT_MAXIMIZED:
      ctx_client_lower_bottom (ctx, ctx_client_id (active));
      break;
    case LAYOUT_FOUR_BY_THREE:
      if (!locked)
        ctx_client_raise_almost_top (ctx, ctx_client_id (active));
      else
        ctx_client_raise_top (ctx, ctx_client_id (active));
      break;
    case LAYOUT_WINDOWS:
      ctx_client_raise_top (ctx, ctx_client_id (active));
      break;
  }

  update_vts_light_mode (ctx);
  update_maximized_size (ctx);

  //terminal_broadcast_message (ctx, "foo!");

  return ctx_client_id (active);
}

int add_tab_argv (Ctx  *ctx, char **argv, int can_launch)
{
  float titlebar_h = ctx_height (ctx)/40;
  //int was_maximized = ctx_client_is_maximized (ctx, ctx_clients_active (ctx));
  int flags = CTX_CLIENT_UI_RESIZABLE |  CTX_CLIENT_TITLEBAR;
  if (can_launch) flags |= CTX_CLIENT_CAN_LAUNCH;

  CtxClient *active = ctx_client_new_argv (ctx, argv, add_x, add_y,
                    ctx_virtual_width(ctx)/2, (ctx_virtual_height (ctx) - titlebar_h)/2,
                    font_size,
                    flags, NULL, NULL);
  
  //VT *vt = ctx_client_vt (active);
  add_y += ctx_virtual_height (ctx) / 20;
  add_x += ctx_virtual_height (ctx) / 20;

  if (add_y + ctx_height(ctx)/2 > ctx_client_max_y_pos (ctx))
  {
    add_y = ctx_client_min_y_pos (ctx);
    add_x -= ctx_height (ctx) / 40 * 4;
  }

  switch (layout_no)
  {
    case LAYOUT_FOUR_BY_THREE:
      if (!locked)
        ctx_client_raise_almost_top (ctx, ctx_client_id (active));
      else
        ctx_client_raise_top (ctx, ctx_client_id (active));
      break;
    case LAYOUT_WINDOWS:
      ctx_client_raise_top (ctx, ctx_client_id (active));
      break;
  }

  ctx_queue_draw (ctx);
  update_vts_light_mode (ctx);
  return ctx_client_id (active);
}

int ctx_clients_tab_to_id (Ctx *ctx, int tab_no);

void swap_front_most (Ctx *ctx)
{
    CtxList *l = ctx_clients (ctx);
    int tab_no = 0;
    int client_count = ctx_list_length (l);
    CtxClient *client = ctx_list_nth_data (l, client_count - 1 - tab_no);
    ctx_client_raise_almost_top (ctx, ctx_client_id (client));
    ctx_queue_draw (ctx);
    ctx_client_focus (ctx, ctx_client_id (client));
}

void switch_to_tab (Ctx *ctx, int tab_no)
{
  if (layout_no == LAYOUT_FOUR_BY_THREE ||
      layout_no == LAYOUT_MAXIMIZED)
  {
    CtxList *l = ctx_clients (ctx);
    int client_count = ctx_list_length (l);

    if (tab_no == 0)
      tab_no = 1;
    else if (tab_no == 1)
    {
      tab_no = 0;
    }

    CtxClient *client = ctx_list_nth_data (l, client_count - 1 - tab_no);
    int id = ctx_client_id (client);
    if (tab_no <= 1)
    {
      if (client)
        ctx_client_focus (ctx, id);
    }
    else
    {
      if (client)
      {
        ctx_client_focus (ctx, id);
        if (layout_no == LAYOUT_FOUR_BY_THREE)
          ctx_client_raise_almost_top (ctx, id);
      }
    }
    ctx_queue_draw (ctx);
  }
}

CtxClient *ctx_client_by_id (Ctx *ctx, int id);

CtxEvent *ctx_event_copy (CtxEvent *event);



int         vt_get_scroll           (VT *vt);
void        vt_set_scroll           (VT *vt, int scroll);


void vt_set_font_size (VT *vt, float font_size);
void vt_recompute_cols (VT *vt);

void set_new_font_size (Ctx *ctx, float new_font_size)
{
  font_size = new_font_size;
  cfg_set_f32 (&ctx_config, "ctx", "font_size", new_font_size, NULL);
  for (CtxList *l = ctx_clients (ctx); l; l = l->next)
  {
    CtxClient *client = l->data;
    VT *vt = ctx_client_vt (client);
    vt_set_font_size (vt, new_font_size); //
    vt_set_line_spacing (vt, line_spacing); //
    vt_set_baseline (vt, baseline); //
    vt_recompute_cols (vt);
  }
  ctx_queue_draw (ctx);
}


static void wm_focus_cb (Ctx *ctx, int focused_id, void *user_data)
{
  ctx_client_raise_top (ctx, focused_id);
}

static void update_maximized_size (Ctx *ctx)
{
  CtxList *clients = ctx_clients (ctx);
  int n_clients = ctx_list_length (clients);

  float tab_spacing = 0;
  if (n_clients > 1)
  {
    float fsy = font_size / ctx_virtual_height (ctx);
    tab_spacing = fsy * titlebar_height;
  }

  if (on_screen_keyboard)
  {
    ctx_clients_maximized_rect (ctx, 0,tab_spacing,
       1, 
       1.0-ctx_osk_height(ctx)/ ctx_virtual_height (ctx) - tab_spacing);
  }
  else
  {
    ctx_clients_maximized_rect (ctx, 0,tab_spacing,1,1-tab_spacing);
  }
}

static void toggle_magnifier (Ctx *ctx)
{
  magnifier_active = !magnifier_active;
  ctx_queue_draw (ctx);
}

static void toggle_magnifier_cb (CtxEvent *event, void *data1, void *data2)
{
  toggle_magnifier (event->ctx);
}



static void toggle_keyboard (Ctx *ctx)
{
  on_screen_keyboard = !on_screen_keyboard;
  update_maximized_size (ctx);
  ctx_queue_draw (ctx);
}

static void toggle_keyboard_cb (CtxEvent *event, void *data1, void *data2)
{
  toggle_keyboard (event->ctx);
}

static void raise_to_top (CtxEvent *event, void *data1, void *data2)
{
  CtxClient *client = data1;
  Ctx *ctx = event->ctx;
  ctx_client_raise_top (ctx, ctx_client_id (client));
  ctx_queue_draw (ctx);
  ctx_client_focus (ctx, ctx_client_id (client));
  event->stop_propagate = 1;
}

static int in_settings = 0;

typedef enum PanelItemType {
  PANEL_ITEM_TYPE_DEFAULT = 0,
  PANEL_ITEM_TYPE_24H_CLOCK,
  PANEL_ITEM_TYPE_ANALOG_CLOCK,
  PANEL_ITEM_TYPE_ORIENTATION,
  PANEL_ITEM_TYPE_FONT_SIZE,
  PANEL_ITEM_TYPE_VOLUME,
  PANEL_ITEM_TYPE_BACKLIGHT,
} PanelItemType;


typedef struct PanelItem {
  int type;
  char *label;
  char *action;
  char *icon;
} PanelItem;

PanelItem panel_items[]=
{
  //{PANEL_ITEM_TYPE_24H_CLOCK, "clock", "new_term", NULL},
  {0, "ctx", "toggle_panel", NULL},
  {0, "new", "new_term", NULL},
  {0, "kb", "kb", NULL},
  {0, "maximize", "maximize", NULL},
  {0, "settings", "settings", NULL},
  {0, "magnifier", "magnifier", NULL},
  {0, "lock", "lock", NULL},
  //{0, "light_mode", "light_mode", NULL},
  //{PANEL_ITEM_TYPE_ORIENTATION, "orientation", "orientation", NULL},
};
void panel_action (Ctx *ctx, const char *action);

void toggle_light_mode (Ctx *ctx)
{
  update_vts_light_mode (ctx);
}

void toggle_light_mode_cb (CtxEvent *e, void *a, void *b)
{
  toggle_light_mode (e->ctx);
}

enum
{
  TAB_KEYBINDINGS = 0,
  TAB_UI,
  TAB_COLORS,
  TAB_FONTS ,
  TAB_INPUT,
  TAB_RENDERING,
  TAB_HELP,
  TAB_ABOUT,
  TAB_LICENSE,
};
static int active_tab = TAB_UI;

int snap_size (Ctx *ctx, void *data);

int check_size (Ctx *ctx, void *data)
{
  float *tmp = data;

  if (ctx_width(ctx) != tmp[0] || tmp[2] < 1)
  {
    ctx_set_size (ctx, tmp[0], tmp[1]+1);
    usleep (1000 * 200);
    ctx_queue_draw (ctx);
    tmp[2] += 1.0f;
    return 0;
  }
  else
    free (data);
  return 0;
}

int snap_size (Ctx *ctx, void *data)
{
  float *tmp = data;
  ctx_set_size (ctx, tmp[0], tmp[1]);
  usleep (1000 * 150); // hack
  ctx_queue_draw (ctx);

  //ctx_free (tmp);
  ctx_add_timeout (ctx, 0, check_size, data);
  return 0;
}

void toggle_fullscreen (Ctx *ctx)
{
  int val = !ctx_get_fullscreen (ctx);
  cfg_set_boolean (&ctx_config, "ctx", "fullscreen", val, NULL);
  if (!val)
    layout_no = LAYOUT_MAXIMIZED;
  else
    layout_no = LAYOUT_FOUR_BY_THREE;

  int w = ctx_client_width (ctx, ctx_clients_active (ctx));
  int h = ctx_client_height (ctx, ctx_clients_active (ctx));

  ctx_set_fullscreen (ctx, val);
  if (!val)
  {
    {
    float *tmp = ctx_malloc (sizeof(float)*3);
    tmp[0] = w+1;
    tmp[1] = h;
    tmp[2] = 0;
    ctx_add_timeout (ctx, 0, snap_size, tmp);
    }
  }
  ctx_queue_draw (ctx);
}

void screenshot_action (Ctx *ctx)
{
    time_t now = time(NULL);
    struct tm *timenow = localtime (&now);
    char path[256];
    if (getenv ("XDG_DATA_HOME"))
    {
      char format[256];
      sprintf (format, "%s/ctx/ctx--%%Y%%m%%d_%%H%%M%%S.png", getenv("XDG_DATA_HOME"));
      strftime (path, sizeof(path), format, timenow);
    }
    else
      strftime (path, sizeof(path), "/tmp/ctx-%Y%m%d_%H%M%S.png", timenow);
    make_ancestors (path);
    ctx_screenshot (ctx, path);
}

void screenshot_action_cb (CtxEvent *event, void *a, void *b)
{
  screenshot_action (event->ctx);
  event->stop_propagate = 1;
}

static void
terminal_handle_event (Ui *ui,
                       Ctx        *ctx,
                       CtxEvent   *ctx_event,
                       const char *event)
{
  int active_id = ctx_clients_active (ctx);
  CtxClient *active = active_id>=0?ctx_client_by_id (ctx, active_id):NULL;
  if (!active)
  {
     printf("no active client\n");
    return;
  }
  //if (active->internal)
  //  return;
  VT *vt = ctx_client_vt (active);

  CtxClient *client = active; //vt_get_client (vt);

  if (!vt)
  {
     ctx_client_add_event (active, ctx_event);
     return;
  }

  ctx_client_lock (client);

  if (locked)
  {
    ctx_client_feed_event (active, ctx_event, event);
    ctx_client_unlock (client);
    return;
  }

  if (panel_focused)
  {
    if (!strcmp (event, "up"))
    {
      panel_focused ++;
      if (panel_focused >=
          (int)(sizeof(panel_items)/sizeof(panel_items[0])))
        panel_focused = sizeof(panel_items)/sizeof(panel_items[0])-1;
    }
    else if (!strcmp (event, "down"))
    {
      panel_focused --;
      if (panel_focused == 0)
        panel_open = 0;
    }
    else if (!strcmp (event, "escape"))
    {
      panel_focused = 0;
    }
    else if (!strcmp (event, "return") ||
             !strcmp (event, "left") ||
             !strcmp (event, "right"))

    {
      panel_action (ctx, panel_items[panel_focused].action);

      panel_focused = 0;
    }
    ctx_queue_draw (ctx);
    ctx_client_unlock (client);
    return;
  }

  
  if (!strcmp (event, "shift-control-d"))
  {
    toggle_light_mode (ctx);
  }
  else if (!strcmp (event, "shift-control-l"))
  {
    toggle_magnifier (ctx);
  }
  else if (!strcmp (event, "F11") ||
      !strcmp (event, "shift-control-f"))
  {
    toggle_fullscreen (ctx);
  }
  else if (!strcmp (event, "shift-control-m"))
  {
    if (layout_no == LAYOUT_MAXIMIZED)
      layout_no = LAYOUT_FOUR_BY_THREE;
    else
      layout_no = LAYOUT_MAXIMIZED;

    ctx_queue_draw (ctx);
  }
  else if (!strcmp (event, "shift-control-k"))
  {
    toggle_keyboard (ctx);
  }
  else if (!strcmp (event, "shift-control-i"))
  {
    screenshot_action (ctx);
  }
  else if (!strcmp (event, "shift-control-p"))
  {
    in_settings = !in_settings;
    panel_open = in_settings;
    ctx_queue_draw (ctx);
  }
  else if (!strcmp (event, "shift-control-h"))
  {
    in_settings = !in_settings;
    panel_open = in_settings;
    if (in_settings)
    {
      active_tab = TAB_HELP;
    }
    ctx_queue_draw (ctx);
  }
  else if (!strcmp (event, "shift-return"))
  {
    ctx_client_feed_event (active, ctx_event, "return");
  }
#if ENABLE_PORTRAIT
  else if (!strcmp (event, "shift-control-r") )
  {
     rotate (ctx);
  }
#endif
  else if (!strcmp (event, "shift-control-v") )
  {
    char *text = ctx_get_clipboard (ctx);
    if (text)
      {
        ctx_client_paste (ctx, active_id, text);
        free (text);
      }
  }
  else if (!strcmp (event, "shift-control-c") && vt)
  {
    char *text = ctx_client_get_selection (ctx, active_id);
    if (text)
      {
        ctx_set_clipboard (ctx, text);
        free (text);
      }
  }
  else if (!strcmp (event, "shift-control-t"))
  {
    if (!terminal_no_new_tab)
      add_tab (ui, ctx, ctx_find_shell_command(), 1);
  }
  else if (!strcmp (event, "shift-control-n") )
  {
    if (!terminal_no_new_tab)
    {
             // XXX : we should also avoid this when running with kms/fb !
    pid_t pid;
    if ( (pid=fork() ) ==0)
      { 
        unsetenv ("CTX_VERSION");
        unsetenv ("CTX_BACKEND");
        execlp (execute_self, execute_self, NULL);
        exit (0);
      }
    }
  }
  else if (!strcmp (event, "alt-1"))   switch_to_tab(ctx, 0);
  else if (!strcmp (event, "alt-2"))   switch_to_tab(ctx, 1);
  else if (!strcmp (event, "alt-3"))   switch_to_tab(ctx, 2);
  else if (!strcmp (event, "alt-4"))   switch_to_tab(ctx, 3);
  else if (!strcmp (event, "alt-5"))   switch_to_tab(ctx, 4);
  else if (!strcmp (event, "alt-6"))   switch_to_tab(ctx, 5);
  else if (!strcmp (event, "alt-7"))   switch_to_tab(ctx, 6);
  else if (!strcmp (event, "alt-8"))   switch_to_tab(ctx, 7);
  else if (!strcmp (event, "alt-9"))   switch_to_tab(ctx, 8);
  else if (!strcmp (event, "alt-0")) {
     // switch_to_tab(ctx, 9);
    CtxEvent dummy = {};
    dummy.ctx = ctx;
    if (second)
      raise_to_top (&dummy, second, NULL);
  }
  else if (!strcmp (event, "shift-control-q") )
  {
    ctx_exit (ctx);
  }
#if 0
  else if (!strcmp (event, "control-tab") ||
           !strcmp (event, "alt-tab"))
  {
     CtxList *l = ctx_clients (ctx);
     int found = 0;
     CtxClient *next = NULL;
     for (; l; l = l->next)
     {
       CtxClient *client = l->data;
       if (found)
       {
         if (!next) next = client;
       }
       if (client == active) found = 1;
     }
     if (!next)
       next = ctx_clients (ctx)->data;
     ctx_client_focus (ctx, ctx_client_id (next));
     ctx_queue_draw (ctx);
  }
  else if (!strcmp (event, "shift-control-tab") ||
           !strcmp (event, "shift-alt-tab"))
  {
     CtxList *l = ctx_clients (ctx);
     CtxClient *prev = NULL;
     if (l->data == active)
     {
       for (; l; l = l->next)
         prev = l->data;
     }
     else
     {
       for (; l; l = l->next)
       {
         CtxClient *client = l->data;
         if (client == active)
           break;
         prev = client;
       }
     }
     ctx_client_focus (ctx, ctx_client_id (prev));
     ctx_queue_draw (ctx);
  }
#endif

  else if (!strcmp (event, "shift-control-right") )
  {
    panel_open = 1;
    panel_focused = 1;
    ctx_queue_draw (ctx);
  }

  else if (!strcmp (event, "shift-control-up") )
  {
    if (vt)
    {
      vt_set_scroll (vt, vt_get_scroll (vt)+1);
      ctx_queue_draw (ctx);
    }
  }
  else if (!strcmp (event, "shift-control-down") )
  {
    if (vt)
    {
      vt_set_scroll (vt, vt_get_scroll (vt)-1);
      ctx_queue_draw (ctx);
    }
  }
  else if (!strcmp (event, "shift-control-w") )
  {
    ctx_client_unlock (client);
    ctx_client_remove (ctx, client);
    ctx_queue_draw (ctx);
    return;
  }
  else if (  !strcmp (event, "control-+") 
           ||!strcmp (event, "shift-control-+") 
        )
  {
  }
  else if (( !strcmp (event, "control-=") ||!strcmp (event, "shift-control-=")))
  {
    int old_rows = ctx_height (ctx) / font_size;
    float new_font_size = ctx_height (ctx) * 1.0f / (old_rows-1.0f);
    float new_rows = ctx_height (ctx) / new_font_size;

    if (old_rows <=4) goto done;

    while (fmodf(new_rows, 1.0f) > 0.2)
    {
      new_font_size /= 1.001;
      new_rows = ctx_height (ctx) / new_font_size;
    }

    set_new_font_size (ctx, new_font_size);
  }
  #if 1
  else if (!strcmp (event, "shift-control-e"))  // change windowing engine
  {
    layout_no++;
    if (layout_no>2)
      layout_no = LAYOUT_WINDOWS;

    switch (layout_no)
    {
      case LAYOUT_MAXIMIZED:
         ctx_set_focus_cb (ctx, NULL, NULL);
         for (CtxList *l = ctx_clients (ctx); l; l = l->next)
         {
           //CtxClient *client = l->data;
           //int id = ctx_client_id (client);
           //ctx_client_maximize (ctx, id);
         }
        break;
      case LAYOUT_FOUR_BY_THREE:
      case LAYOUT_WINDOWS:
         ctx_set_focus_cb (ctx, wm_focus_cb, NULL);
         for (CtxList *l = ctx_clients (ctx); l; l = l->next)
         {
           CtxClient *client = l->data;
           int id = ctx_client_id (client);
           ctx_client_unmaximize (ctx, id);
           int w = ctx_virtual_width (ctx)/2  - font_size;
           int h = ctx_virtual_height (ctx)/2 - font_size * 2;
           ctx_client_resize (ctx, id, w, h);
         }
        break;
    }

    ctx_queue_draw (ctx);
  }
  #endif
  else if (!strcmp (event, "control--"))
  {
    // TODO : make these computations based on terminal size..
    //        not ctx size
    int old_rows = roundf(ctx_height (ctx) / font_size);
    float new_font_size = ctx_height (ctx) * 1.0f / (old_rows+1.0f);
    float new_rows = ctx_height (ctx) / new_font_size;
    //fprintf (stderr, "-(%s)", event);
    while (floorf(new_rows)==old_rows)//fmodf(new_rows, 1.0f) > 0.05)
    {
      new_font_size /= 1.001;
      new_rows = ctx_height (ctx) / new_font_size;
    }

    if (font_size < 8) font_size = 8;
    set_new_font_size (ctx, new_font_size);
  }
  else if (!strcmp (event, "shift-control-/") )
  {
    help_elapsed = 0.0f;
    ctx_queue_draw (ctx);
  }
#if 0
  else if (!strcmp (event, "shift-control-s") )
  {
    CtxEvent dummy = {};
    dummy.ctx = ctx;
    if (second)
      raise_to_top (&dummy, second, NULL);
  }
#endif
  else if (!strncmp (event, "shift-control-", 14))
  {
  }
  else if (!strncmp (event, "alt-control-", 12))
  {
  }
  else if (!strncmp (event, "shift-alt-control-", 18))
  {
  }
  else
  {
//    if (ctx_event->type == CTX_KEY_DOWN && !strcmp (ctx_event->string, "alt"))
//    {
//      fprintf (stderr, "%s\n", event);
 //     exit (1);
 //   }
    ctx_client_feed_event (active, ctx_event, event);
  }
  done:
  ctx_client_unlock (client);
}

static float consume_float (char **argv, int *i)
{
 float ret = 0.0;
 if (!argv[*i+1])
 {
   fprintf (stderr, "error parsing arguments\n");
   exit (2);
 }
 ret = atof (argv[*i+1]);
 (*i)++;
  int j;
  for (j = *i; argv[j]; j++)
  {
    argv[j-2] = argv[j];
  }
  argv[j-2]=0;
  (*i)-=2;
  return ret;
}
// good font size for 80col:
//
// 4 6 8 10 12 14 16 18 20 2
extern int on_screen_keyboard;
extern float osk_width; // in percent
void ctx_osk_draw (Ctx *ctx);

#if GNU_C
static int malloc_trim_cb (Ctx *ctx, void *data)
{
  malloc_trim (64*1024);
  return 1;
}
#endif

int saved_cursor = -1;

static void terminal_show_cursor (CtxEvent *event, void *userdata, void *userdata2)
{
  if (saved_cursor == -1) return;
  ctx_set_cursor (event->ctx, CTX_CURSOR_ARROW);
  saved_cursor = -1;
  ctx_queue_draw (event->ctx);
}

static void terminal_hide_cursor (CtxEvent *event, void *userdata, void *userdata2)
{
  if (saved_cursor == -1)
  {
    saved_cursor = ctx_get_cursor (event->ctx);
    ctx_set_cursor (event->ctx, CTX_CURSOR_NONE);
  }
}

static int alt_pressed = 0;

static void terminal_key_any (CtxEvent *event, void *userdata, void *userdata2)
{
  Ui *ui = userdata;
  if (event->type == CTX_KEY_PRESS &&
      event->string &&
      !strcmp (event->string, "resize-event"))
  {
    Ctx *ctx = event->ctx;
    cfg_set_int (&ctx_config, "ctx", "width", ctx_virtual_width (ctx), NULL);
    cfg_set_int (&ctx_config, "ctx", "height", ctx_virtual_height (ctx), NULL);
    ctx_queue_draw (ctx);
  }
  else
  {
    switch (event->type)
    {
      case CTX_KEY_PRESS:
        terminal_handle_event (ui, ctx, event, event->string);
        if (cfg_boolean (&ctx_config, "", "auto_hide_mouse"))
          terminal_hide_cursor (event, userdata, userdata2);
        break;

      case CTX_TEXT_INPUT:
        { char buf[1024];
          snprintf (buf, 1023, " %s", event->string);
          terminal_handle_event (ui, ctx, event, buf);
        }
        break;

      case CTX_KEY_UP:
        if (!strcmp (event->string, "alt"))
        {
          alt_pressed = 0;
          ctx_queue_draw (event->ctx);
        }

        { char buf[1024];
          snprintf (buf, sizeof(buf)-1, "ku %i %i", event->scan, event->state);
          terminal_handle_event (ui, ctx, event, buf);
        }
        break;
      case CTX_KEY_DOWN:
        { char buf[1024];
          snprintf (buf, sizeof(buf)-1, "kd %i %i", event->scan, event->state);
          terminal_handle_event (ui, ctx, event, buf);
        }
        if (!strcmp (event->string, "alt"))
        {
          alt_pressed = 1;
          ctx_queue_draw (event->ctx);
        }
        break;
      default:
        break;
    }
  }
}

extern int _ctx_enable_hash_cache;

static void icon_remove_tab (Ctx *ctx, float x, float y, float w, float h)
{
  ctx_save (ctx);
  ctx_translate (ctx, (x + 0.5 * w), (y + 0.5 * w));
  ctx_rotate (ctx, M_PI/4);
  ctx_translate (ctx, -0.5 * w, -0.5 * w);
  ctx_rectangle (ctx, 0.4 * w, 0.1 * h, 0.2 * w, 0.8 * h);
  ctx_rectangle (ctx, 0.1 * w, 0.4 * h, 0.8 * w, 0.2 * h);
  ctx_restore (ctx);
}

static void icon (Ctx *ctx, float x, float y, float w, float h, const char *name)
{
  ctx_save (ctx);
  ctx_text_align (ctx, CTX_TEXT_ALIGN_START);
  ctx_translate (ctx, x, y);
  ctx_scale (ctx, w, h);
  const char *str = get_icon (name);
  if (str)
  {
    ctx_parse (ctx, str);
  }
  else
  {
    ctx_move_to (ctx, x, y);
    ctx_text (ctx, name);
  }
  ctx_restore (ctx);
}

static void lock (Ctx *ctx)
{
  char *command = ctx_strdup_printf ("touch %s", ctx_term_lock_path ());
  if (system (command))
  {
  }
  free (command);
  ctx_queue_draw (ctx);
}

static void lock_cb (CtxEvent *event, void *a, void *b)
{
  lock (event->ctx);
}

static inline float panel_width (Ctx *ctx)
{
  return (ctx_virtual_width (ctx) / 4);
}

static inline float panel_height (Ctx *ctx)
{
  return (ctx_virtual_height (ctx) / 6);
}


void panel_action (Ctx *ctx, const char *action)
{
  ctx_queue_draw (ctx);
  if (!strcmp (action, "toggle_panel"))
  {
    if (in_settings)
    {
      in_settings = 0;
      panel_open = 0;
    }
    else
    {
      panel_open = !panel_open;
    }
    return;
  }
  else if (!strcmp (action, "magnifier"))
  {
    toggle_magnifier (ctx);
  }
  else if (!strcmp (action, "kb"))
  {
    toggle_keyboard (ctx);
  }
  else if (!strcmp (action, "light_mode"))
  {
    update_vts_light_mode (ctx);
  }
  else if (!strcmp (action, "lock"))
  {
    in_settings = 0;
    lock (ctx);
  }
  else if (!strcmp (action, "maximize"))
  {
    if (layout_no == LAYOUT_MAXIMIZED)
      layout_no = LAYOUT_FOUR_BY_THREE;
    else
      layout_no = LAYOUT_MAXIMIZED;
    ctx_queue_draw (ctx);
  }
  else if (!strcmp (action, "orientation"))
  {
    rotate (ctx);
  }
  else if (!strcmp (action, "settings"))
  {
    in_settings = !in_settings;
  }
  else if (!strcmp (action, "font+"))
  {
    int old_rows = ctx_height (ctx) / font_size;
    float new_font_size = ctx_height (ctx) * 1.0f / (old_rows-1.0f);
    float new_rows = ctx_height (ctx) / new_font_size;

    if (old_rows <=4) return;

    while (fmodf(new_rows, 1.0f) > 0.2)
    {
      new_font_size /= 1.001;
      new_rows = ctx_height (ctx) / new_font_size;
    }

    set_new_font_size (ctx, new_font_size);
    return;
  }
  else if (!strcmp (action, "font-"))
  {
    // TODO : make these computations based on terminal size..
    //        not ctx size
    int old_rows = roundf(ctx_height (ctx) / font_size);
    float new_font_size = ctx_height (ctx) * 1.0f / (old_rows+1.0f);
    float new_rows = ctx_height (ctx) / new_font_size;
    //fprintf (stderr, "-(%s)", event);
    while (floorf(new_rows)==old_rows)//fmodf(new_rows, 1.0f) > 0.05)
    {
      new_font_size /= 1.001;
      new_rows = ctx_height (ctx) / new_font_size;
    }

    if (font_size < 8) font_size = 8;
    set_new_font_size (ctx, new_font_size);
    return;
  }
  else if (!strcmp (action, "new_term"))
  {
    if (!terminal_no_new_tab)
      add_tab (ui, ctx, ctx_find_shell_command(), 1);
  }
  if (!in_settings)
    panel_open = 0;
}


void panel_action_cb (CtxEvent *event, void *a, void *b)
{
  const char *action = a;
  Ctx *ctx = event->ctx;
  event->stop_propagate=1;
  if (event->type == CTX_MOTION)
  {
    ctx_queue_draw (ctx);
    return;
  }

  if (event->type != CTX_DRAG_RELEASE)
    return;

  panel_action (ctx, action);
}

const char *icons[]={
  "lock", "translate 0 0.1 M 0.3 0.4 arc 0.5 0.3 0.2 3.14 6.3 0 L 0.7 0.4 lineWidth 0.1 stroke rectangle 0.2 0.4 0.6 0.4 fill ",
  "kb", "save scale 0.8 0.8 translate 0.15 0.1 "
        "lineWidth 0.02 roundRectangle "
        "0.0 0.3 0.09 0.09 0.02 "
        "0.0 0.3 0.09 0.09 0.02 "
        "0.1 0.3 0.09 0.09 0.02 "
        "0.2 0.3 0.09 0.09 0.02 "
        "0.3 0.3 0.09 0.09 0.02 "
        "0.4 0.3 0.09 0.09 0.02 "
        "0.5 0.3 0.09 0.09 0.02 "
        "0.6 0.3 0.09 0.09 0.02 "
        "0.7 0.3 0.09 0.09 0.02 "
        "0.8 0.3 0.09 0.09 0.02 "

        "0.03 0.4 0.09 0.09 0.02 "
        "0.13 0.4 0.09 0.09 0.02 "
        "0.23 0.4 0.09 0.09 0.02 "
        "0.33 0.4 0.09 0.09 0.02 "
        "0.43 0.4 0.09 0.09 0.02 "
        "0.53 0.4 0.09 0.09 0.02 "
        "0.63 0.4 0.09 0.09 0.02 "
        "0.73 0.4 0.09 0.09 0.02 "
        "0.83 0.4 0.09 0.19 0.02 "

        "0.03 0.5 0.09 0.09 0.02 "
        "0.16 0.5 0.09 0.09 0.02 "
        "0.26 0.5 0.09 0.09 0.02 "
        "0.36 0.5 0.09 0.09 0.02 "
        "0.46 0.5 0.09 0.09 0.02 "
        "0.56 0.5 0.09 0.09 0.02 "
        "0.66 0.5 0.09 0.09 0.02 "
        "0.76 0.5 0.09 0.09 0.02 "

        "0.1 0.6 0.09 0.09 0.02 "
        "0.2 0.6 0.09 0.09 0.02 "
        "0.3 0.6 0.09 0.09 0.02 "
        "0.4 0.6 0.09 0.09 0.02 "
        "0.5 0.6 0.09 0.09 0.02 "
        "0.6 0.6 0.09 0.09 0.02 "
        "0.7 0.6 0.09 0.09 0.02 "
        "0.8 0.6 0.09 0.09 0.02 "

        "0.0 0.7 0.19 0.09 0.02 "
        "0.2 0.7 0.6  0.09 0.02 "

        " fill restore"
          ,
  "zoom", "arc 0.5 0.5 0.3 0.0 6.4 0 arc 0.5 0.5 0.2 6.4 0.0 1 fill",
  "1up", "roundRectangle 0.0 0.1 0.8 0.8 0.05 lineWidth .04 stroke "
         "roundRectangle 0.8 0.1 0.2 0.2 0.05 lineWidth .04 stroke "
         "roundRectangle 0.8 0.35 0.2 0.2 0.05 lineWidth .04 stroke ",
  "2up", "roundRectangle 0.0 0.35 0.5 0.5 0.05 lineWidth .04 stroke "
         "roundRectangle 0.52 0.35 0.5 0.5 0.05 lineWidth .04 stroke "
         "roundRectangle 0.0 0.1 0.2 0.2 0.05 lineWidth .04 stroke "
         "roundRectangle 0.2 0.1 0.2 0.2 0.05 lineWidth .04 stroke "
         "roundRectangle 0.4 0.1 0.2 0.2 0.05 lineWidth .04 stroke ",
  "font+", "fontSize 0.6 M 0.2, 0.7 text 'A+'",
  "font-", "fontSize 0.6 M 0.2, 0.7 text 'A-'",
  "new", "roundRectangle 0.2 0.2 0.6 0.6 0.05 lineWidth .04 stroke fontSize 0.3 M 0.3 0.6 text '$'",

  "settings", 

     "lineWidth 0.05 "
     "M 0.1 0.3 L 0.9 0.3 stroke "
     "roundRectangle 0.2 0.25 0.1 0.1 0.05 fill " 
     "translate 0 0.2"
     "M 0.1 0.3 L 0.9 0.3 stroke "
     "roundRectangle 0.8 0.25 0.1 0.1 0.05 fill " 
     "translate 0 0.2"
     "M 0.1 0.3 L 0.9 0.3 stroke "
     "roundRectangle 0.3 0.25 0.1 0.1 0.05 fill " 

             ,

  "orientation", "lineWidth 0.1 arc 0.5 0.5 0.3 3.14 8.3 0 M 0.2 0.5 l0 0.15.16-.16-.16.16-.16-.16 stroke",

  "close", "lineWidth 0.10 M 0.2 0.2 L 0.8 0.8 M 0.8 0.2 L 0.2 0.8 stroke",

  "switch",
"lineCap round lineJoin round "
"lineWidth 0.10 M 0.2 0.45 m 0.15 0.15 l -0.15 -0.15 l 0.15 -0.15 l -0.15 0.15 l 0.6 0.0 l -0.15 0.15 l 0.15 -0.15 l -0.15 -0.15 l 0.15 0.15 stroke ",

  "up",
  "lineWidth 0.06 lineJoin round"
          " M 0.5 0.3 m -0.20 0.20 l 0.20 -0.20 l 0.20 0.20 m -0.20 -0.20 l 0 0.4"
          " stroke",

  "down",
  "lineWidth 0.06 lineJoin round"
          " M 0.5 0.7 m -0.20 -0.20 l 0.20 0.20 l 0.20 -0.20 m -0.20 0.20 l 0 -0.4"
          " stroke",

  "right",
  "lineWidth 0.06 lineJoin round"
          " M 0.7 0.5 m -0.20 -0.20 l 0.20 0.20 l -0.20 0.20 m 0.20 -0.20 l -0.4 0"
          " stroke",

  "tab",
  "lineWidth 0.06 lineJoin round"
          " M 0.7 0.5 m -0.20 -0.20 l 0.20 0.20 l -0.20 0.20 m 0.20 -0.20 l -0.4 0"
          " M 0.8 0.2 l 0 0.6 "
          " stroke",

  "left",
  "lineWidth 0.06 lineJoin round"
          " M 0.3 0.5 m 0.20 -0.20 l -0.20 0.20 l 0.20 0.20 m -0.20 -0.20 l 0.4 0"
          " stroke",

  "backspace",
  "lineWidth 0.06 lineJoin round"
          " M 0.2 0.5 l 0.2 0.2 l 0.3 0 l 0 -0.4 l -0.3 0 l -0.2 0.2 z "
          " M 0.4 0.4 l 0.2 0.2 M 0.6 0.4 l -0.2 0.2"
          " stroke",

  "return",
  "lineWidth 0.06 lineJoin round"
          " M 0.3 0.5 m 0.20 -0.20 l -0.20 0.20 l 0.20 0.20 m -0.20 -0.20 l 0.4 0l 0.0 -0.4"
          " stroke",

  "light_mode",
  "arc 0.5 0.5 0.35 0.0 6.3 0 z "
  "arc 0.5 0.5 0.3 0.2 3.34 0 z "
  "fillRule evenodd "
  "fill ",

  "magnifier",
  "translate 0 0.1 "
  "lineWidth 0.08 "
    " arc 0.5 0.4 0.3 0.0 6.3 0 M 0.7 0.6 l 0.2 0.2 stroke ",

  "maximize",
  "lineWidth 0.10 lineJoin round"
          " M 0.5 0.1 m -0.16 0.16 l 0.16 -0.16 l 0.16 0.16 m -0.16 -0.16 l 0 0.3"
          " M 0.9 0.5 m -0.16 -0.16 l 0.16 0.16 l -0.16 0.16 m 0.16 -0.16 l -0.3 0"
          " M 0.1 0.5 m 0.16 -0.16 l -0.16 0.16 l 0.16 0.16 m -0.16 -0.16 l 0.3 0"

          " M 0.5 0.9 m -0.16 -0.16 l 0.16 0.16 l 0.16 -0.16 m -0.16 0.16 l 0 -0.3"
          " stroke",


  "unmaximize", "lineWidth 0.10 "
          " M 0.5 0.6 m -0.16 0.16 l 0.16 -0.16 l 0.16 0.16 m -0.16 -0.16 l 0 0.3"
          " M 0.4 0.5 m -0.16 -0.16 l 0.16 0.16 l -0.16 0.16 m 0.16 -0.16 l -0.3 0"
          " M 0.6 0.5 m 0.16 -0.16 l -0.16 0.16 l 0.16 0.16 m -0.16 -0.16 l 0.3 0"

          " M 0.5 0.4 m -0.16 -0.16 l 0.16 0.16 l 0.16 -0.16 m -0.16 0.16 l 0 -0.3"
          " stroke",

  NULL, NULL
};

const char *get_icon (const char *label)
{
  for (int n = 0; icons[n*2]; n++)
  if (!strcmp (icons[n*2], label))
    return icons[n*2+1];

  return NULL;
}


int ctx_client_is_active_tab (Ctx *ctx, CtxClient *client);
void draw_panel (Ctx *ctx)
{
  float panel_w = panel_width (ctx);
  //float panel_h = panel_height (ctx);

  ctx_save (ctx);
  int item_w = panel_w/4;
  int item_h = panel_w/4;
  ctx_translate (ctx, ctx_virtual_width (ctx) - item_w, 0);
  ctx_reset_path (ctx);

  int n_items = sizeof(panel_items)/sizeof(panel_items[0]);

  if (panel_open)
  {
    ctx_rectangle (ctx, 0, ctx_virtual_height(ctx)-item_h*n_items, item_w, item_h * n_items);
    ui_color_base (ui, 0.4, 0.2);
    ctx_fill (ctx);
  }

  int row = 0;
  for (int i = 0 ; i<n_items && (i==0 || panel_open);i++)
  {
    int col = 0;

    ctx_save (ctx);
    ctx_translate (ctx, col * item_w,
                        ctx_virtual_height (ctx) - (row+1) * item_h);
    ctx_reset_path (ctx);
    ctx_rectangle (ctx, 0, 0, item_w, item_h);
    ctx_listen (ctx, CTX_MOTION|CTX_DRAG, panel_action_cb, panel_items[i].action, NULL);
    ctx_reset_path (ctx);
    ctx_move_to (ctx, (item_w-2)/2, (item_h-2)/2);

    float x = ctx_pointer_x (ctx) - (ctx_virtual_width (ctx) - item_w);
    float y = ctx_virtual_height (ctx) - ctx_pointer_y (ctx);


    char sbuf[80]="00:00";
    char *buf = panel_items[i].label;
    if (panel_items[i].type == PANEL_ITEM_TYPE_24H_CLOCK)
      buf = sbuf;

    if (!strcmp (buf, "maximize"))
    {
      if (layout_no == LAYOUT_MAXIMIZED)
        buf = "unmaximize";
    }

    if (
        (x >= col * item_w  &&
        x < col * item_w + item_w &&
        y >= row * item_h &&
        y < row * item_h + item_h) ||
        (panel_focused && panel_focused == row))
    {
       ctx_save (ctx);
       ctx_reset_path (ctx);
       ctx_round_rectangle (ctx, item_w * 0.05, item_w * 0.05,
                       item_w * 0.9, item_h * 0.9, em * 0.9);
       //ctx_line_width (ctx, item_w * 0.05);
       //if (panel_focused && panel_focused == row)
         STYLE_TEXT;
       //else
       //STYLE_WIDGET_BASE;
       ctx_fill (ctx);
       ctx_restore (ctx);
    }

    if ((x >= col * item_w  &&
        x < col * item_w + item_w &&
        y >= row * item_h &&
        y < row * item_h + item_h) ||
        (panel_focused && panel_focused == row))

      STYLE_WIDGET_BASE;
    else
      if ( 
        (!strcmp (buf, "kb") && (on_screen_keyboard!=0)) ||
        (!strcmp (buf, "magnifier") && (magnifier_active!=0)) ||
        (!strcmp (buf, "settings") && (in_settings!=0)) 
        )
    {
      STYLE_TEXT_ACTIVE;
    }
    else
      STYLE_TEXT;

    const char *icon = get_icon (buf);

    if (!strcmp (buf, "ctx"))
    {
      ctx_translate (ctx, (item_w-item_h)/2, 0);
      if (panel_open)
      {
        ctx_logo (ctx, item_h/2, item_h/2, item_h * 0.9);
      }
      else
      {
        ui_color_ink (ui, 0.3, -0.2);
        ctx_line_width (ctx, 4.0);
        ctx_logo_stroke (ctx, item_h/2, item_h/2, item_h * 0.9);
        ctx_stroke (ctx);
      }
    }
    else if (icon)
    {
      ctx_translate (ctx, (item_w-item_h)/2, 0);
      ctx_scale (ctx, item_h/1.0, item_h/1.0);
      ctx_reset_path (ctx);
      ctx_parse (ctx, icon);
    }
    else if (buf)
    {
      ctx_text_align (ctx, CTX_TEXT_ALIGN_CENTER);
      ctx_text_baseline (ctx, CTX_TEXT_BASELINE_MIDDLE);
      ctx_text (ctx, buf);
    }
    ctx_restore (ctx);
    row ++;
  }


  ctx_restore (ctx);
}
static char *set_title = NULL;
void vt_audio_task (VT *vt, int click);

void
terminal_update_title (const char *title)
{
  if (!title)
    title="NULL";
  if (set_title)
  {
    if (!strcmp (set_title, title))
    {
      return;
    }
    free (set_title);
  }
  set_title = strdup (title);
  ctx_windowtitle (ctx, set_title);
}

int ctx_input_pending (Ctx *ctx, int timeout);

static uint64_t path_modtime (const char *path)
{
  struct stat attrib;
  stat(path, &attrib);
  return attrib.st_mtime;
}

int commandline_argv_start = 0;

void ctx_client_draw (Ctx *ctx, CtxClient *client, float x, float y);

typedef struct OverviewPos {
  int id;
  CtxClient *client;
  float x0;
  float y0;
  float w0;
  float h0;
  float x1;
  float y1;
  float scale0;  /// is always 1.0f
  float scale1;

} OverviewPos;

static OverviewPos  *opos           = NULL;
static int           opos_count = 0;

static void overview_cleanup (Ctx *ctx)
{
  if (opos)
  {
    ctx_free (opos);
    opos = NULL;
  }
  opos_count = 0;
}

static void overview_init (Ctx *ctx)
{
}





void ctx_client_use_images (Ctx *ctx, CtxClient *client);

void ctx_term_lock_screen (Ctx *ctx)
{
  //ctx_rgb (ctx, 1,0.0,0);
  //ctx_paint (ctx);
  ctx_rgb (ctx, 1,1,1);

  CtxList *clients = ctx_clients (ctx);

  CtxClient *last = NULL;

  for (CtxList *l = clients; l; l = l->next)
  {
    CtxClient *client = l->data;
    if (l->next)
      ctx_client_use_images (ctx, client);
    last = client;
  }

  if (last)
    ctx_client_draw (ctx, last, 0, 0);
}

static int clients_draw (Ctx *ctx)
{
  CtxList *clients = ctx_clients (ctx);
  int n_clients = ctx_list_length (clients);

  if (n_clients == 1) {
    CtxClient *client = clients->data;
    int flags = ctx_client_flags (client);
    if (client && flag_is_set(flags, CTX_CLIENT_MAXIMIZED))
    {
      ctx_client_draw (ctx, client, 0, 0);
      return 0;
    }
  }
#if 0
  float screen_width = ctx_width (ctx) - 3 * em;
  float screen_height = ctx_height (ctx);
#endif
  int active_id = ctx_clients_active (ctx);
  CtxClient *active = active_id>=0?ctx_client_by_id (ctx, active_id):NULL;

  for (CtxList *l = clients; l; l = l->next)
  {
    CtxClient *client = l->data;
    int flags = ctx_client_flags (client);
    if (flag_is_set(flags, CTX_CLIENT_MAXIMIZED))
    {
       if (client == active)
        ctx_client_draw (ctx, client, 0, 0);
      else
        ctx_client_use_images (ctx, client);
    }
  }

  return 0;
}

static void load_palette(void)
{
  uint8_t rgba[4];
  cfg_rgba_full (&ctx_config, "palette", "color15", rgba, "#ffffff", NULL);
  vt_set_palette(NULL, 15, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color14", rgba, "#1de1ed", NULL);
  vt_set_palette(NULL, 14, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color13", rgba, "#d66fed", NULL);
  vt_set_palette(NULL, 13, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color12", rgba, "#82b4ed", NULL);
  vt_set_palette(NULL, 12, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color11", rgba, "#e9d808", NULL);
  vt_set_palette(NULL, 11, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color10", rgba, "#99edba", NULL);
  vt_set_palette(NULL, 10, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color9", rgba, "#edac82", NULL);
  vt_set_palette(NULL, 9, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color8", rgba, "#6f6f6f", NULL);
  vt_set_palette(NULL,8, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color7", rgba, "#c3c3c3", NULL);
  vt_set_palette(NULL,7, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color6", rgba, "#3b6bb1", NULL);
  vt_set_palette(NULL,6, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color5", rgba, "#ab4adf", NULL);
  vt_set_palette(NULL,5, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color4", rgba, "#2424ed", NULL);
  vt_set_palette(NULL,4, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color3", rgba, "#878453", NULL);
  vt_set_palette(NULL,3, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color2", rgba, "#4aa08b", NULL);
  vt_set_palette(NULL,2, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color1", rgba, "#a02929", NULL);
  vt_set_palette(NULL,1, rgba[0], rgba[1], rgba[2]);
  cfg_rgba_full (&ctx_config, "palette", "color0", rgba, "#000000", NULL);
  vt_set_palette(NULL,0, rgba[0], rgba[1], rgba[2]);
}

static void load_fonts(Ctx *ctx)
{
#if CTX_HARFBUZZ
  char *fonts[]={ "Mono Bold Italic", 
                  "Mono Italic",
                  "Mono Bold",
                  "Mono",
                  "Regular",
                  NULL, };
  for (int i = 0; fonts[i]; i++)
  {
    char *font_path = cfg_string_full (&ctx_config, "fonts", fonts[i], "", NULL);
    if (font_path)
    {
      if (font_path[0])
        ctx_load_font_file (ctx, fonts[i], font_path);
      else
        ctx_load_font_file (ctx, fonts[i], NULL);
      free (font_path);
    }
  }
#endif
  float old_line_spacing = line_spacing;
  float old_baseline = baseline;

  font_size = cfg_f32_full (&ctx_config, "ctx", "font_size", 19.5f, "font size in pixels");
  line_spacing = cfg_f32_full (&ctx_config, "fonts", "line_spacing", 1.0f, NULL);
  baseline = cfg_f32_full (&ctx_config, "fonts", "baseline", 0.8f, NULL);
  if (line_spacing != old_line_spacing ||
      baseline != old_baseline)
  {
    set_new_font_size (ctx, font_size);
  }
}


static int no_save_config = 0;
extern int ctx_vt_enable_ligatures;

static void load_config (Ctx *ctx)
{
  if(0)load_palette ();
  if(ctx)
  load_fonts (ctx);

  cfg_boolean_full (&ctx_config, "", "auto_hide_mouse", 0, NULL);
  ctx_vt_enable_ligatures = cfg_boolean_full (&ctx_config, "vt", "coding_ligatures", -1,
"enable_ligatures makes consequetive runs of the same light color to be\n"
"# shaped by harfbuzz; triggering coding font ligatures for things like:\n"
"#  != </> /> == === <- -> => ~~ 0x000 ");


  if (ctx)
  {
  if (cfg_boolean_full (&ctx_config, "ctx", "internal_clipboard", -1,
"internal_clipboard disables use of system clipboard in SDL backend, thus making\n"
"# other applications not able to snoop on the contents.")>0)
    ctx_internal_clipboard (ctx, 1);
  else
    ctx_internal_clipboard (ctx, 0);
  }

  on_screen_keyboard = cfg_boolean_full (&ctx_config, "ctx", "on_screen_keyboard", 0, "");

  animation_duration = cfg_f32_full (&ctx_config, "style", "animation_duration", 0.175f, NULL);

  cfg_boolean (&ctx_config, "", "auto_hide_mouse");

  cfg_f32_full (&ctx_config, "fonts", "baseline", 0.78f, NULL);
  cfg_f32_full (&ctx_config, "style", "window_radius", 0.8f, NULL);
  cfg_f32_full (&ctx_config, "style", "border_width", 0.0f, NULL);
  cfg_f32_full (&ctx_config, "style", "margin", 0.2f, NULL);
  cfg_f32_full (&ctx_config, "style", "resize_border_width", 0.4f, NULL);

  titlebar_height = cfg_f32_full (&ctx_config, "style", "titlebar_height", 1.5f, NULL);

 if (ctx)
 {
   cfg_ctx_color_full (&ctx_config, "style", "window_close", ctx, "#a00", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "panel_item", ctx, "#8887", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "panel_item_hovered", ctx, "#aaa", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "window_shadow", ctx, "#00000077", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "wallpaper", ctx, "#7", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "warm", ctx, "#f00", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "cool", ctx, "#0af", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "brightest", ctx, "#f", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "darkest", ctx, "#0", NULL);
   cfg_ctx_color_full (&ctx_config, "style", "debug", ctx, "#ff000077", NULL);

   update_maximized_size (ctx);
 }

 cfg_f32_full (&ctx_config, "style", "contrast", STYLE_DARK_CONTRAST, NULL);
 cfg_f32_full (&ctx_config, "style", "brightness", STYLE_DARK_BRIGHTNESS, NULL);
 cfg_f32_full (&ctx_config, "style", "saturation", STYLE_DARK_SATURATION, NULL);
 cfg_f32_full (&ctx_config, "style", "cool-hue", 0.58f, NULL);
 cfg_f32_full (&ctx_config, "style", "warm-hue", 0.11f, NULL);
 cfg_f32_full (&ctx_config, "style", "base-saturation", 50.0f, NULL);
 cfg_f32_full (&ctx_config, "style", "background-temperature", 0.0f, NULL);
 cfg_f32_full (&ctx_config, "style", "focused-saturation", 0.0f, NULL);
 cfg_f32_full (&ctx_config, "style", "border_shadow_x", 0.0f, NULL);
 cfg_f32_full (&ctx_config, "style", "border_shadow_y", 0.0f, NULL);

 animation_duration = cfg_f32_full (&ctx_config, "style", "animation_duration", 0.175f, NULL);
}

void sync_config (void)
{
  cfg_set_boolean (&ctx_config, "ctx", "on_screen_keyboard", on_screen_keyboard, NULL);
  cfg_set_boolean (&ctx_config, "vt", "coding_ligatures", ctx_vt_enable_ligatures, NULL);

  cfg_set_f32 (&ctx_config, "style", "animation_duration", animation_duration, NULL);
  cfg_set_f32 (&ctx_config, "ctx", "line_spacing", line_spacing, NULL);
  cfg_set_f32 (&ctx_config, "ctx", "baseline", baseline, NULL);
  cfg_set_f32 (&ctx_config, "osk", "width", osk_width, NULL);
}


void ctx_client_titlebar_drag (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_n (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_w (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_e (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_s (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_nw (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_ne (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_sw (CtxEvent *event, void *data, void *data2);
void ctx_client_resize_se (CtxEvent *event, void *data, void *data2);
void ctx_client_close (CtxEvent *event, void *data, void *data2);



typedef enum {
  DECOR_NONE      = 0,
  DECOR_TITLEBAR  = (1<<0),
  DECOR_FOCUSED   = (1<<1), 
  DECOR_RESIZABLE = (1<<2), 
  DECOR_MOVABLE   = (1<<3),
  DECOR_CLOSE     = (1<<4),
  DECOR_MAXIMIZE  = (1<<5)
} decor_flag;

static void client_pick (CtxEvent *event, void *data1, void *data2)
{
  ctx_client_focus (event->ctx, ctx_client_id (data1));
}


static int tab_actions_open = 0;

static int tab_actions_active = 0;

float tile_dim = 128;

static void tab_actions_cb (CtxEvent *event,
                            void *a, void *b)
{
  CtxClient *client = a;
  Ctx *ctx = event->ctx;
  float tab_actions_width = tile_dim * 3;
  int client_id = ctx_client_id (client);

  switch (event->type)
  {
     case CTX_DRAG_PRESS:
       if (tab_actions_open == 0)
       {
         tab_actions_open = 1;
         ctx_queue_draw (ctx);
         return;
       }
       /* FALLTHROUGH */
     case CTX_MOTION:
     case CTX_DRAG_MOTION:
     case CTX_DRAG_RELEASE:

      float ex = event->x - (ctx_client_width (ctx, client_id) - tab_actions_width);

       //fprintf (stderr, "%f %f %f\n", event->x, event->y, tile_dim);
       if (ex < tile_dim * 3 && ex > 0 &&
           event->y > font_size && event->y < tile_dim)
       {
         tab_actions_active = (int)(ex / tile_dim);
         //printf ("  %i %f\n", tab_actions_active, event->x/tile_dim);
         if (event->type == CTX_DRAG_RELEASE)
         {
           switch (tab_actions_active)
           {
              case 0: // swap front
                raise_to_top (event, second, b);
                break;
              case 1:
                if (layout_no == LAYOUT_MAXIMIZED)
                  layout_no = LAYOUT_FOUR_BY_THREE;
                else
                  layout_no = LAYOUT_MAXIMIZED;
                break;
              case 2:
                ctx_client_close (event, client, b);
                break;
           }
           tab_actions_open = 0;
         }
       }
       else
       {
         if (event->type == CTX_DRAG_RELEASE)
         {
           if (event->y > font_size)
             tab_actions_open = 0;
         }
         if (event->type == CTX_DRAG_PRESS)
         {
           if (event->y < font_size)
             tab_actions_open = !tab_actions_open;
         }
         tab_actions_active = -1;
       }
       break;
    break;
    default:
  }
  ctx_event_stop_propagate (event);
  ctx_queue_draw (ctx);
}

void draw_tab_actions (Ctx *ctx, CtxClient *client, float client_x, float client_y)
{
  float tab_actions_width = tile_dim * 3;
  int client_id = ctx_client_id (client);
  ctx_save (ctx);
  ctx_text_align (ctx, CTX_TEXT_ALIGN_START);

  STYLE_WIDGET_BASE;
  ctx_reset_path (ctx);
  float y = client_y;
  float x = client_x + ctx_client_width(ctx, client_id) - tab_actions_width;
  ctx_rectangle (ctx, x, y, tab_actions_width, tile_dim + em * 0.51);
  ctx_save (ctx);
  ctx_translate (ctx, client_x, client_y);
  ctx_listen (ctx, CTX_MOTION|CTX_DRAG, tab_actions_cb, client, NULL);
  ctx_restore (ctx);
  ctx_fill (ctx);
  STYLE_WIDGET_BORDER;
  ctx_rectangle (ctx, x, y, tab_actions_width, tile_dim + em * 0.51);
  ctx_stroke (ctx);
  y+=em;ctx_move_to (ctx, x, y);

  int tile_no = 0;

  if ((tile_no == tab_actions_active))
    STYLE_TEXT_ACTIVE;
  else
    STYLE_TEXT;
  icon (ctx, x + tile_no * tile_dim, client_y, tile_dim, tile_dim, "switch");
  tile_no ++;

  if ((tile_no == tab_actions_active))
    STYLE_TEXT_ACTIVE;
  else
    STYLE_TEXT;
  icon (ctx, x + tile_no * tile_dim, client_y, tile_dim, tile_dim,
  "maximize");
  tile_no ++;

  if ((tile_no == tab_actions_active))
    STYLE_TEXT_ACTIVE;
  else
    STYLE_TEXT;
  icon (ctx, x + tile_no * tile_dim, client_y, tile_dim, tile_dim,
  "close");

  ctx_restore (ctx);
}

void decorate_client (Ctx *ctx, CtxClient *client, int client_x, int client_y, float scale, int flags)
{
  int is_focused = (flags & DECOR_FOCUSED) != 0;
  int id = ctx_client_id (client);
  //int flags = ctx_client_flags (client);

  float rbw = cfg_f32 (&ctx_config, "style", "resize_border_width") * em;
  float bw  = cfg_f32 (&ctx_config, "style", "border_width") * em;
  float sx  = cfg_f32 (&ctx_config, "style", "border_shadow_x") * em;
  float sy  = cfg_f32 (&ctx_config, "style", "border_shadow_y") * em;

  //int client_x = ctx_client_x (ctx, id);
  //int client_y = ctx_client_y (ctx, id);
  int client_width = ctx_client_width (ctx, id);
  int client_height = ctx_client_height (ctx, id);
  float window_radius = cfg_f32 (&ctx_config, "style", "window_radius") * em;

  tile_dim = em * 4;

  if (is_focused)
    STYLE_HOVERED;
  else
    STYLE_WIDGET_BASE;

  ctx_reset_path (ctx);
  if (flags & DECOR_TITLEBAR)
  {
    ctx_rectangle (ctx, client_x - bw, client_y - titlebar_height * em,
                      client_width + bw * 2, titlebar_height * em);

    if (flags & DECOR_MOVABLE)
    {
      ctx_listen (ctx, CTX_DRAG, ctx_client_titlebar_drag, client, NULL);
      ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_ALL);
    }
    else
    if (scale == 1.0f)
    {
      ctx_listen (ctx, CTX_DRAG, client_pick, client, NULL);
    }

    if (window_radius > 0.01)
    {
      ctx_reset_path (ctx);
      ctx_arc (ctx, client_x - bw + window_radius, client_y - (titlebar_height * em) + window_radius, window_radius, M_PI, M_PI * 1.5f, 0);
      ctx_arc (ctx, client_x + bw + client_width - window_radius, client_y - (titlebar_height * em) + window_radius, window_radius, M_PI * 1.5f, M_PI * 2, 0);
      ctx_line_to (ctx, client_x + bw + client_width, client_y+1);
      ctx_line_to (ctx, client_x - bw, client_y+1);
    }

    ctx_fill (ctx);

    if (is_focused  && layout_no == LAYOUT_WINDOWS)
    {
    int pos = 1;
    if (flags & DECOR_CLOSE)
    {
      ctx_rectangle (ctx, client_x + client_width + bw - font_size * pos,
                     client_y - (titlebar_height * em) * 0.5f - em/2 , 
                     font_size, font_size);
      ctx_listen (ctx, CTX_CLICK, ctx_client_close, client, NULL);
      ctx_listen_set_cursor (ctx, CTX_CURSOR_CROSSHAIR);
      //ctx_fill (ctx);
      ctx_reset_path (ctx);

      ctx_move_to (ctx, client_x + client_width + bw - font_size * pos,
                     client_y - (titlebar_height * em)/2 - em/2 + 0.8 * em);
      ctx_save (ctx);
      STYLE_TEXT_INTERACTIVE;
      ctx_text (ctx, "X");
      ctx_restore (ctx);
      pos ++;
    }
    }
  }

  if (bw > 0.0f)
  {
    ctx_rectangle (ctx, client_x - bw, client_y,
                        bw, client_height);
    ctx_fill (ctx);
    ctx_rectangle (ctx, client_x + client_width, client_y,
                        bw, client_height);
    ctx_fill (ctx);
    if (window_radius > 0.01)
    {
      ctx_reset_path (ctx);
      ctx_arc (ctx, client_x - bw + window_radius, client_y + client_height + bw - window_radius,
               window_radius, M_PI, M_PI/2, 1);
      ctx_arc (ctx, client_x + bw + client_width - window_radius,
               client_y + client_height + bw - window_radius,
               window_radius,
               M_PI/2, 0, 1);
      ctx_line_to (ctx, client_x + bw + client_width, client_y + client_height);
      ctx_line_to (ctx, client_x - bw, client_y + client_height);
    }
    else
    {
      ctx_rectangle (ctx, client_x - bw, client_y + client_height,
                          client_width + bw * 2, bw);
    }
    ctx_fill (ctx);
    if ((flags & DECOR_TITLEBAR)==0)
    {
      if (window_radius > 0.01)
      {
        ctx_reset_path (ctx);
        ctx_arc (ctx, client_x - bw + window_radius, client_y - bw + window_radius, window_radius, M_PI, M_PI * 1.5f, 0);
        ctx_arc (ctx, client_x + bw + client_width - window_radius, client_y - bw + window_radius, window_radius, M_PI * 1.5f, M_PI * 2, 0);
        ctx_line_to (ctx, client_x + bw + client_width, client_y);
        ctx_line_to (ctx, client_x - bw, client_y);
      }
      else
      {
        ctx_rectangle (ctx, client_x - bw, client_y - bw,
                          client_width + bw * 2, bw);
      }

      ctx_fill (ctx);
    }
  }

  // shadow - only for active window
  if (fabsf(sx) > 0.0f && is_focused)
  {
    //color ("window_shadow");

    if (sx > 0)
    {
      ctx_rectangle (ctx, client_x + client_width + bw, client_y + sy - (titlebar_height * em),
                           sx, client_height + (titlebar_height * em) + bw);
      ctx_fill (ctx);
      ctx_rectangle (ctx, client_x + sx, client_y + client_height + bw,
                           client_width - sx + bw, sy);
      ctx_fill (ctx);
    }
  }

  if (rbw > 0.0f && is_focused && ((flags & DECOR_RESIZABLE) != 0))
  {
    ctx_rgba(ctx, 1,0,0,0.5);
    ctx_rectangle (ctx, client_x - rbw, client_y,
                        rbw, client_height);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_w, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_W);
    ctx_reset_path (ctx);
    ctx_rectangle (ctx, client_x + client_width, client_y,
                        rbw, client_height);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_e, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_E);
    ctx_reset_path (ctx);

    ctx_rectangle (ctx, client_x, client_y + client_height,
                        client_width, rbw);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_s, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_S);
    ctx_reset_path (ctx);

    ctx_rectangle (ctx, client_x - rbw, client_y + client_height,
                        rbw, rbw);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_sw, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_SW);
    ctx_reset_path (ctx);

    ctx_rectangle (ctx, client_x + client_width, client_y + client_height,
                        rbw, rbw);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_se, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_SE);
    ctx_reset_path (ctx);

    ctx_rectangle (ctx, client_x, client_y - (titlebar_height * em) - rbw,
                        client_width, rbw);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_n, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_N);
    ctx_reset_path (ctx);

    ctx_rectangle (ctx, client_x - rbw, client_y - (titlebar_height * em) - rbw,
                        rbw, rbw);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_nw, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_NW);
    ctx_reset_path (ctx);

    ctx_rectangle (ctx, client_x + client_width, client_y - (titlebar_height * em) - rbw,
                        rbw, rbw);
    ctx_listen (ctx, CTX_DRAG, ctx_client_resize_ne, client, NULL);
    ctx_listen_set_cursor (ctx, CTX_CURSOR_RESIZE_NE);
    ctx_reset_path (ctx);
  }


  if (flags & DECOR_TITLEBAR)
  {
        // draw title
        ctx_save (ctx);
        if (is_focused)
          STYLE_WIDGET_BASE;
        else
          STYLE_TEXT;

        const char *title = ctx_client_get_title (ctx, id);
        if (layout_no == LAYOUT_WINDOWS)
        {
          ctx_text_align (ctx, CTX_TEXT_ALIGN_CENTER);
          ctx_move_to (ctx, client_x + client_width/2, client_y - (titlebar_height * em) * 0.2f);
        }
        else
        {
          //ctx_text_align (ctx, CTX_TEXT_ALIGN_START);
          //ctx_move_to (ctx, client_x, client_y - (titlebar_height * em) * 0.2f);
          ctx_text_align (ctx, CTX_TEXT_ALIGN_END);
          ctx_move_to (ctx, client_x + client_width - window_radius/2, client_y - (titlebar_height * em) * 0.2f);

        }
        
        ctx_text (ctx, title);
        ctx_restore (ctx);



        if (is_focused)
        {
          float tab_actions_width = tile_dim * 3;
          if (tab_actions_width < ctx_text_width (ctx, title))
          {
            tab_actions_width  = ctx_text_width (ctx, title);
          }
          ctx_reset_path (ctx);
          ctx_rectangle (ctx, client_x + client_width-tab_actions_width, client_y - (titlebar_height * 2 * em),
                              tab_actions_width, titlebar_height * 2 * em);
          ctx_save (ctx);
          ctx_listen (ctx, CTX_DRAG, tab_actions_cb, client, NULL);
          ctx_reset_path (ctx);
          ctx_restore (ctx);
#if 0
          ctx_rectangle (ctx, client_x + client_width-tab_actions_width, client_y - (titlebar_height * em * 0.6) ,
                              tab_actions_width, titlebar_height * em * 0.2);
          ctx_save (ctx);
          ui_color_ink (ui, 0.5);
          ctx_fill (ctx);
          ctx_restore (ctx);
#endif
        }
  }
}

void window_manager (Ctx *ctx, float delta_s)
{
  cfg_ctx_color (&ctx_config, "style", "wallpaper", ctx);
  ctx_paint(ctx);

  CtxList *clients = ctx_clients (ctx);
  //int n_clients = ctx_list_length (clients);

  int active_id = ctx_clients_active (ctx);
  CtxClient *active = active_id>=0?ctx_client_by_id (ctx, active_id):NULL;

  for (CtxList *l = clients; l; l = l->next)
  {
    CtxClient *client = l->data;
    int id = ctx_client_id (client);
    int flags = ctx_client_flags (client);
    if (flag_is_set(flags, CTX_CLIENT_MAXIMIZED))
    {
      if (client == active)
        ctx_client_draw (ctx, client, ctx_client_x(ctx, id), ctx_client_y(ctx, id));
      else
        ctx_client_use_images (ctx, client);

    } else
    {
        int client_x = ctx_client_x (ctx, id);
        int client_y = ctx_client_y (ctx, id);

        ctx_client_draw (ctx, client, client_x, client_y);
        decorate_client (ctx, client, client_x, client_y, 1.0f,
          DECOR_TITLEBAR |
          DECOR_CLOSE |
          DECOR_MAXIMIZE |
          DECOR_RESIZABLE | 
          DECOR_MOVABLE | 
          ((active == client) * DECOR_FOCUSED) );
        if (tab_actions_open && (active == client))
          draw_tab_actions (ctx, client, client_x, client_y);

    }
  }

  draw_panel (ctx);
}

void maximized_tabs_cb (CtxEvent *event, void *a, void *b)
{
  Ctx *ctx = event->ctx;
  CtxList *l = ctx_clients (ctx);
  int n_clients = ctx_list_length (l);

  int active_client = (int)(event->x / ((ctx_virtual_width (ctx)/n_clients)));

  if (event->type == CTX_DRAG_PRESS)
  {
    switch_to_tab (ctx, active_client);
    ctx_queue_draw (ctx);
  }

}


void
maximized (Ctx *ctx, float delta_s)
{
  CtxList *clients = ctx_clients (ctx);
  int client_count = ctx_list_length (clients);

  int active_id = ctx_clients_active (ctx);
  CtxClient *active = active_id>=0?ctx_client_by_id (ctx, active_id):NULL;

  cfg_ctx_color (&ctx_config, "style", "wallpaper", ctx);
  ctx_paint(ctx);
  ctx_reset_path (ctx);
  ctx_save (ctx);
  int no = 0;
  {
    for (CtxList *l = clients; l; l = l->next, no++)
    {
      CtxClient *client = l->data;
      int is_active = client == active;
      if (is_active)
      {
         float scale = 1.0f;

         int flags = ctx_client_flags (client);
         if (!flag_is_set(flags, CTX_CLIENT_MAXIMIZED))
           ctx_client_maximize (ctx, ctx_client_id (client));
         float ytranslate = font_size * titlebar_height;
         if (client_count <= 1)
           ytranslate = 0;
         ctx_save (ctx); // a

#if 0
//       float cursor_x = ctx_vt_cursor_x (client) * vt_cw (ctx_client_vt (client));
         float cursor_y = ctx_vt_cursor_y (client) * vt_ch (ctx_client_vt (client));
  static float real_translate = 0.0f;
  float target_translate = 0.0f;

  if (on_screen_keyboard)
  {
    if (cursor_y > ctx_height (ctx) * 0.96f)
    {
      target_translate = 0; // something like activating a search or other globalish - view
                            // perhaps dim keyboard instead?
    }
    else
    if (cursor_y > ctx_height (ctx) * 0.666f)
    {
      target_translate = (-ctx_height (ctx) * 0.333f)  *   (cursor_y - ctx_height(ctx) * 0.666) / ((ctx_height(ctx) * 0.33))   ;
    }
    else
    {
      target_translate = 0;
    }
  }
  if (delta_s < 0.1)
    for (float i = delta_s; i > 0.0; i -= 0.01)
      real_translate = real_translate * 0.92 + target_translate * 0.08;
  else
    real_translate = real_translate * 0.9 + target_translate * 0.1;
  if (fabs (real_translate- target_translate) > .5f)
  {
    ctx_queue_draw (ctx);
  }
  else
    real_translate = target_translate;
  ctx_translate (ctx, 0, real_translate);
#endif

         ctx_save (ctx); // b
          ctx_translate (ctx, 0, ytranslate);
          ctx_scale (ctx, scale, scale);
          ctx_client_draw (ctx, client, 0, 0);
          if (tab_actions_open)
            draw_tab_actions (ctx, client, 0, 0);
         ctx_restore (ctx); // b
         if (client_count > 1)
         {
           float available = ctx_virtual_width (ctx);
           float per_tab = available / client_count;
           
           int active_id = ctx_clients_active (ctx);
           int pos = 0;
           ctx_save (ctx); // c
           ctx_rectangle (ctx, 0, 0, available, font_size * titlebar_height);
           ui_color_base (ui, 0.6, 0.2);
           ctx_listen (ctx, CTX_DRAG, maximized_tabs_cb, NULL, NULL);
           ctx_fill (ctx);

           ctx_restore (ctx); // c
           for (CtxList *l = clients; l; l = l->next, pos++)
           {
             CtxClient *client = l->data;
             int is_active = 0;
             if (ctx_client_id (client) == active_id)
               is_active = 1;
             int tpos = pos;
             if  (tpos == client_count - 1)
               tpos = client_count - 2;
             else if (tpos == client_count - 2)
               tpos = client_count - 1;

             ctx_save (ctx); // d
             ctx_translate (ctx, (client_count - tpos - 1) * per_tab, 0);

             if (is_active)
             {
               ctx_rectangle (ctx, 0, 0, per_tab, font_size * titlebar_height);
               ui_color_base (ui, 1.0, -0.3);

               ctx_fill (ctx);
             }
             else if (tpos != 0)
             {
               ctx_rectangle (ctx, per_tab, font_size * titlebar_height * 0.2,
                  font_size*0.1, font_size * titlebar_height * 0.6);
               ui_color_base (ui, 1.0, 0.0);
               ctx_fill (ctx);
             }

             if (is_active)
             {
               ui_color_ink (ui, 1.0, -0.6);
             }
             else
             {
               ui_color_ink (ui, 0.7, 0);
             }

             const char *title = ctx_client_title (client);
             float title_width = ctx_text_width (ctx, title);

             if (title_width > (per_tab - font_size))
             {
               ctx_font_size (ctx, font_size * ((per_tab - font_size)/title_width));
             }
             ctx_move_to (ctx, per_tab/2, font_size);
             ctx_text_align (ctx, CTX_TEXT_ALIGN_CENTER);
             ctx_text (ctx, ctx_client_title (client));
             ctx_restore (ctx); // d
           }
         }
         ctx_restore (ctx); // a
      }
      else
      {
        ctx_client_use_images (ctx, client);
      }
    }
  }

  draw_panel (ctx);
  ctx_restore (ctx);
}

static void raise_to_second (CtxEvent *event, void *data1, void *data2)
{
  CtxClient *client = data1;
  Ctx *ctx = event->ctx;
  ctx_client_raise_almost_top (ctx, ctx_client_id (client));
  ctx_queue_draw (ctx);
  ctx_client_focus (ctx, ctx_client_id (client));
  event->stop_propagate = 1;
}

typedef struct 
{
  CtxClient *client;
  int   id;
  float x;
  float y;
  float scale;
  float opacity;
  float start_x;
  float start_y;
  float start_scale;
  float start_opacity;
  float target_x;
  float target_y;
  float target_scale;
  float target_opacity;
  float deadline;
  float elapsed;
  uint64_t last_alive;
} AnimatedPos;

#define MAX_CLIENTS 32

AnimatedPos anim_state[MAX_CLIENTS]={{0,}};

static AnimatedPos *get_state(Ctx *ctx, CtxClient *client)
{
  uint64_t time = ctx_ticks ();
  int id = ctx_client_id (client);
  int oldest = -1;
  uint64_t oldest_time = 1;
  oldest_time <<= 63;
  for (int i = 0; i < MAX_CLIENTS; i++)
  {
    if (anim_state[i].id == id)
    {
      anim_state[i].last_alive = time;
      return &anim_state[i];
    }
    if (anim_state[i].last_alive < oldest_time)
    {
      oldest_time = anim_state[i].last_alive;
      oldest = i;
    }
  }
  for (int i = 0; i < MAX_CLIENTS; i++)
  {
    if (anim_state[i].client == NULL || (i == oldest))
    {
      anim_state[i].client = client;
      anim_state[i].id = id;
      anim_state[i].x = ctx_virtual_width (ctx)*0.25;
      anim_state[i].y = ctx_virtual_height (ctx)*0.66;
      anim_state[i].last_alive = time;
      return &anim_state[i];
    }
  }
  return &anim_state[MAX_CLIENTS-1];
}

static float easeInOutCubic(float x){
return x < 0.5f ? 4 * x * x * x : 1 - powf(-2 * x + 2, 3.0f) / 2.0;
}

float border_width = 0.1f;

typedef struct layout_info {
  float scale;
  float x;
  float y;
} layout_info;

#define LAYOUT_MAX  20
#define LAYOUT_BASE 2
#define LAYOUT_SIZE (LAYOUT_MAX-LAYOUT_BASE)

layout_info two_up[LAYOUT_SIZE][LAYOUT_SIZE]=
{
  {{1.0,0,0}},
  {{1.0,0,0}},
  {{1.0,0,0}, {1.0, 1, 0}},
  {{1.0,0,0}, {1.0, 1, 0}, {1.0, 2, 0}},
  {{1.0,0,0}, {1.0, 1, 0}, {1.0, 2, 0}, {0.5, 3,0.5}},
  {{1.0,0,0}, {1.0, 1, 0}, {1.0, 2, 0}, {0.5, 3,0.0}, {0.5, 3.0, 0.5}},
  {{1.0,0,0}, {1.0, 1, 0}, {1.0, 2, 0}, {0.5, 3,0.0}, {0.5, 3.0, 0.5}, {0.5, 3.5, 0.0}},
  {{1.0,0,0}, {1.0, 1, 0}, {0.5, 2, 0}, {0.5, 2,0.5}, {0.5, 2.5, 0}, {0.5, 2.5, 0.5}, {0.5, 3.0, 0.0} }, 
  {{1.0,0,0}, {1.0, 1, 0}, {0.5, 2, 0.5}, {0.5, 2,0.0}, {0.5, 2.5, 0.5}, {0.5, 2.5, 0.0}, {0.5, 3.0, 0.5}, {0.5, 3.0, 0.0}},
  {{1.0,0,0}, {1.0, 1, 0}, {0.5, 2, 0.0}, {0.5, 2,0.5}, {0.5, 2.5, 0.0}, {0.5, 2.5, 0.5}, {0.5, 3.0, 0.0}, {0.5, 3.0, 0.5},
   {0.5, 3.5, 0.0}},


  {{1.0,0,0}, {0.5, 1, 0}, {0.5, 1, 0.5}, {0.5, 1.5,0.0}, {0.5, 1.5, 0.5}, {0.5, 2.0, 0.0}, {0.5, 2.0, 0.5}, {0.5, 2.5, 0.0},
   {0.5, 2.5, 0.5}, {0.5, 3.0, 0.0}},

  {{1.0,0,0}, {0.5, 1, 0}, {0.5, 1, 0.5}, {0.5, 1.5,0.5}, {0.5, 1.5, 0.0}, {0.5, 2.0, 0.5}, {0.5, 2.0, 0.0}, {0.5, 2.5, 0.5},
   {0.5, 2.5, 0.0}, {0.5, 3.0, 0.5}, {0.5, 3.0, 0.0}},
  {{1.0,0,0}, {0.5, 1, 0}, {0.5, 1, 0.5}, {0.5, 1.5,0.0}, {0.5, 1.5, 0.5}, {0.5, 2.0, 0.0}, {0.5, 2.0, 0.5}, {0.5, 2.5, 0.0},
   {0.5, 2.5, 0.5}, {0.5, 3.0, 0.0}, {0.5, 3.0, 0.5}, {0.5, 3.5, 0.0}},


  {{0.1, 0, 0},
   {0.1, 0.1, 0},
   {0.1, 0.2, 0},
   {0.1, 0.3, 0},
   {0.1, 0.4, 0},
   {0.1, 0.5, 0},
   {0.1, 0.5, 0},
   {0.1, 0.6, 0},
   {0.1, 0.7, 0},
   {0.1, 0.8, 0},
   {0.1, 0.9, 0},
   {0.1, 0, 0.2},
   {0.1, 0.1, 0.2},
   {0.1, 0.2, 0.2},
   {0.1, 0.3, 0.2},
   {0.1, 0.4, 0.2},
   {0.1, 0.5, 0.2},
   {0.1, 0.5, 0.2}},

};

layout_info two_up_osk[LAYOUT_SIZE][LAYOUT_SIZE]=
{
  {{1.0,0,0}},
  {{1.0,0,0}},
  {{1.0,0,0}, {1.0, 3, 0}},
  {{1.0,0,0}, {0.5, 3, 0}, {0.5, 3.5, 0}},
  {{1.0,0,0}, {0.5, 3, 0}, {0.5, 3.5, 0}, {0.5, 3, 0.5}},
  {{1.0,0,0}, {0.5, 3, 0}, {0.5, 3.5, 0}, {0.25, 3, 0.5}, {0.25, 3.25, 0.5}},
  {{1.0,0,0}, {0.5, 3, 0}, {0.5, 3.5, 0},
   {0.25, 3.25, 0.5},
   {0.25, 3.0, 0.5},
   {0.25, 3.0, 0.75}},
  {{1.0,0,0}, {0.5, 3, 0}, {0.5, 3.5, 0},
   {0.25, 3, 0.5},
   {0.25, 3.25, 0.5},
   {0.25, 3.0, 0.75},
   {0.25, 3.25, 0.75}},
  {{1.0,0,0}, {0.5, 3, 0}, {0.5, 3.5, 0}, {0.25, 3, 0.5},
   {0.25, 3.25, 0.5},
   {0.25, 3.5, 0.5},
   {0.25, 3.0, 0.75},
   {0.25, 3.25, 0.75}},
  {{1.0,0,0}, {0.5, 3, 0},
   {0.5, 3.5, 0},
   {0.25, 3.0,  0.5},
   {0.25, 3.25, 0.5},
   {0.25, 3.5,  0.5},
   {0.25, 3.0,  0.75},
   {0.25, 3.25, 0.75},
   {0.25, 3.5,  0.75}},



  {{1.0,0,0}, {0.5, 3, 0},
   {0.5, 3.5, 0},
   {0.25, 3, 0.5},
   {0.25, 3.25, 0.5},
   {0.25, 3.5, 0.5},
   {0.25, 3.75, 0.5},
   {0.25, 3.0, 0.75},
   {0.25, 3.25, 0.75},
   {0.25, 3.5, 0.75},
  },




};



static void four_by_three_layout (Ctx    *ctx,
                                  int     width,
                                  int     height,
                                  int     focused,
                                  int     no,
                                  int     client_count,
                                  float   small_part,
                                  float   small_scale,
                                  float  *ret_x,
                                  float  *ret_y,
                                  float  *ret_scale,
                                  float  *ret_opacity)
{
  float border = cfg_f32 (&ctx_config, "style", "border_width") * em;
  float margin = cfg_f32 (&ctx_config, "style", "margin") * em;

  float big_y;
  float small_y;

  big_y = titlebar_height * em + margin;
  small_y = big_y + height + titlebar_height * em / 3;

  *ret_opacity = 0.0f;


  if (is_portrait(ctx))
  {
    *ret_opacity = 1.0f;
    *ret_x = 0.0f;
    *ret_scale = (ctx_virtual_width(ctx)*1.0f) / (width + border * 2);///width;//ctx_height(ctx);
    *ret_y = titlebar_height * em * *ret_scale + (titlebar_height + height+ border + margin) * no * *ret_scale;
    return;
  }


    if (no == client_count-1)
    {
      if (client_count == 1)
      {
        *ret_x = (ctx_virtual_width(ctx)-width)/2 - border - margin;
        *ret_y = big_y;
        //*ret_scale *= 1.5f;
      }
      else
      {
        *ret_x = border + margin;
        *ret_y = big_y;
      }
      *ret_opacity = 1.0f;
    }
    else if (no == client_count-2)
    {
      *ret_x = ctx_virtual_width(ctx) - width - border - margin;
      *ret_y = big_y;
      *ret_opacity = 1.0f;
    }
    else if (client_count < 14)
    {
      int lno = client_count - 2;
      int pos = client_count - 3 - no;

      if (on_screen_keyboard)
      {
      *ret_scale = 0.5 * two_up_osk[lno][pos].scale;
      *ret_x = margin + border + (width + border * 2 + margin ) * 0.5 * two_up_osk[lno][pos].x;
      *ret_y = small_y + margin + titlebar_height * em * *ret_scale + (height + titlebar_height * em + border + margin) * 0.5 *  two_up_osk[lno][pos].y;
      }
      else
      {
      *ret_scale = 0.5 * two_up[lno][pos].scale;
      *ret_x = margin + border + (width + border * 2 + margin ) * 0.5 * two_up[lno][pos].x;
      *ret_y = small_y + margin + titlebar_height * em * *ret_scale + (height + titlebar_height * em + border + margin) * 0.5 *  two_up[lno][pos].y;
      }

      *ret_opacity = 1.0f;
      return;
    }
    else if (client_count < 19)
    {

    #if 1
    if (client_count < 16 && no == client_count-3)
    {
      *ret_x = (ctx_virtual_width(ctx) / 2) * (0) * small_scale + border/2 + margin/2;
      *ret_y = small_y;
      *ret_scale = small_scale;
    }
    else if (client_count<13 && no == client_count-4)
    {
      *ret_x = (ctx_virtual_width(ctx) / 2) * (1) * small_scale + border/2 + margin/2;
      *ret_y = small_y;
      *ret_scale = small_scale;
    }
    else if (client_count<10 && no == client_count-5)
    {
      *ret_x = (ctx_virtual_width(ctx) / 2) * (2) * small_scale + border/2 + margin/2;
      *ret_y = small_y;
      *ret_scale = small_scale;
    }
    else if (client_count<7 && no == client_count-6)
    {
      *ret_x = (ctx_virtual_width(ctx) / 2) * (3) * small_scale + border/2 + margin/2;
      *ret_y = small_y;
      *ret_scale = small_scale;
    }
    else
    {
      int base = 3;
      int row, col;


      if (client_count >=16)
      {
        if ((client_count % 2))
          client_count ++;
       
        base = 0;
        row = (client_count-3-no) % 2;
        col = (client_count-3-no) / 2;
      }
      else if (client_count >=13)
      {
        if (!(client_count % 2))
          client_count ++;
       
        base = 1;
        row = (client_count-4-no) % 2;
        col = (client_count-4-no) / 2;
      }
      else if (client_count >=10)
      {
        if ((client_count % 2))
          client_count ++;

        base = 2;
        row = (client_count-5-no) % 2;
        col = (client_count-5-no) / 2;
      }
        else
      {
        if (!(client_count % 2))
          client_count ++;
        base = 3;
        row = (client_count-6-no) % 2;
        col = (client_count-6-no) / 2;
      }
      *ret_x = margin + (ctx_virtual_width(ctx) / 2) * (base) * small_scale + ((border*2+margin+width)*small_scale/2) * col;
      *ret_y = margin + ((height+margin+border+titlebar_height*em) * (small_scale/2) * row);
      *ret_scale = small_scale/2;
    }

      *ret_opacity = 1.0f;
//    *ret_y += panel_height (ctx);
    }
    else 
    {
      float rows = 2;
      if (client_count < 7)
        rows = 1;
      else if (client_count < 9)
        rows = 1.5;
      else if (client_count < 19)
        rows = 2;
      else if (client_count < 30)
        rows = 3;
      
      small_scale /= rows;
      int cols = ctx_virtual_width (ctx) / (width * small_scale);
      int irows = ((client_count-2 + (cols-1)) / cols);
      int i = (client_count-2-1-no);

      int col = i % cols;
      int row = i / cols;

      //col = cols - col - 1;
      row = irows - row - 1;

      *ret_x = (ctx_virtual_width(ctx) / 2) * (col) * small_scale + font_size * 0.5;
      *ret_y = small_y + (height * small_scale + font_size) * row ;
      *ret_scale = small_scale;
      *ret_opacity = 1.0f;
      //*ret_y += panel_height (ctx);
    }
    #endif
}

float vt_cw (VT *vt);
float vt_ch (VT *vt);

void osk_button (Ctx *ctx)
{
  ctx_reset_path (ctx);
  ctx_rectangle (ctx, 0,//ctx_virtual_width (ctx) - font_size * 2.5,
  ctx_virtual_height(ctx) - font_size * 2.5
    , font_size * 3, font_size * 3);
    ctx_listen (ctx, CTX_CLICK, toggle_keyboard_cb, NULL, NULL);
    ui_color_ink_alpha (ui, 0.5, 0.2, 0.3);
  ctx_reset_path (ctx);
}

void four_by_three (Ctx *ctx, float delta_s)
{
  border_width  = cfg_f32 (&ctx_config, "style", "border_width");

  ctx_save (ctx);
  cfg_ctx_color (&ctx_config, "style", "wallpaper", ctx);

  ctx_paint(ctx);

  float em = ctx_get_font_size (ctx);
  CtxList *clients = ctx_clients (ctx);
  int client_count = ctx_list_length (clients);


  second = NULL;
  for (CtxList *l = clients; l; l = l->next)
  {
    if (l->next && l->next->next == NULL)
      {
        second = l->data;
      }
  }

  int no = 0;

  for (CtxList *l = clients; l; l = l->next, no++)
  {
    CtxClient *client = l->data;
    int id = ctx_client_id (client);
    int is_maximized = ctx_client_is_maximized (ctx, id);
    if (is_maximized)
      ctx_client_unmaximize (ctx, id);
  }

  int active_id = ctx_clients_active (ctx);
  CtxClient *active = active_id>=0?ctx_client_by_id (ctx, active_id):NULL;

  float small_scale = 0.5f;
  float small_part  = 0.333f;

  float border = cfg_f32 (&ctx_config, "style", "border_width") * font_size;
  float margin = cfg_f32 (&ctx_config, "style", "margin") * font_size;

  int height = ctx_height (ctx) * (1.0f-small_part) - titlebar_height * font_size - border - margin * 2;
  int width  = ctx_width (ctx) / 2 - border * 2 - margin * 2;


  no = 0;
  for (CtxList *l = clients; l; l = l->next, no++)
  {
    CtxClient *client = l->data;

    if (l->next && l->next->next == NULL)
      {
        client = l->next->data;
      }
    else if (l->next == NULL && second)
      {
        client = second;
      }


    int id = ctx_client_id (client);
    //int flags = ctx_client_flags (client);
    AnimatedPos *astate = get_state(ctx, client);

    //int client_x = ctx_client_x (ctx, id);
    //int client_y = ctx_client_y (ctx, id);
    float client_x;
    float client_y;

    if (ctx_client_width (ctx, id) != width ||
        ctx_client_height (ctx, id) != height)
    {
      ctx_client_resize (ctx, id, width, height);
    }

    float scale = 1.0f;
    float opacity = 1.0f;

    four_by_three_layout (ctx, width, height, active == client, no, client_count, small_part, small_scale, &client_x, &client_y, &scale, &opacity);

    if (astate->target_x != client_x ||
        astate->target_y != client_y ||
        astate->target_scale != scale ||
        astate->target_opacity != opacity)
    {
      astate->start_x = astate->x;
      astate->start_y = astate->y;
      astate->start_scale = astate->scale;
      astate->start_opacity = astate->opacity;
      astate->target_x = client_x;
      astate->target_y = client_y;
      astate->target_scale = scale;
      astate->target_opacity = opacity;
      astate->deadline = animation_duration;
      astate->elapsed = -delta_s;
    }

    if (astate->elapsed > astate->deadline)
    {
      astate->scale = astate->start_scale = astate->target_scale;
      astate->x = astate->start_x = astate->target_x;
      astate->y = astate->start_y = astate->target_y;
      astate->opacity = astate->start_opacity = astate->target_opacity;
    }
    else
    {
      //float t = easeInOutCubic (astate->elapsed / astate->deadline);
      float t = (astate->elapsed / astate->deadline);
      if (t < 0)
        t = 0.0f;
      astate->x     = astate->start_x     * (1.0f - t) + t * astate->target_x;
      astate->y     = astate->start_y     * (1.0f - t) + t * astate->target_y;
      astate->scale = astate->start_scale * (1.0f - t) + t * astate->target_scale;
      astate->opacity = astate->start_opacity * (1.0f - t) + t * astate->target_opacity;
      ctx_queue_draw (ctx);
    }
    astate->elapsed += delta_s;
    
    client_x = astate->x;
    client_y = astate->y;
    scale = astate->scale;

    if (astate->opacity > 0.0f)
    {
      ctx_save (ctx);
      if (astate->opacity != 1.0f)
      {
        ctx_global_alpha (ctx, astate->opacity);
      }
      ctx_translate (ctx, client_x, client_y);
      ctx_scale (ctx, scale, scale);
      ctx_client_draw (ctx, client, 0, 0);

      if (alt_pressed && (active != client))
      {
        int tab_no = client_count - no;
        if (tab_no <= 9)
        {
          ctx_save (ctx);

          ctx_rectangle (ctx, 0, 0,
                            em * 8, em * 8);
          ui_color_base (ui, 0.8f, 0.3);
          ctx_fill (ctx);

          ui_color_ink (ui, 0.7, -0.1);
          ctx_move_to (ctx, em * 4, 0 + em*7);
          ctx_font_size (ctx, em * 8);
          char buf[64];
          snprintf (buf, 63, "%i", tab_no);
          ctx_text_align (ctx, CTX_TEXT_ALIGN_CENTER);
          ctx_text (ctx, buf);

          ctx_restore (ctx);
        }
      }

      decorate_client (ctx, client, 0, 0, scale,
         DECOR_TITLEBAR |
         DECOR_CLOSE |
         //DECOR_MOVABLE |  // causes focus to work on titlebars
         DECOR_MAXIMIZE |
         (active == client) * DECOR_FOCUSED);
        if (tab_actions_open && (active == client))
          draw_tab_actions (ctx, client, 0, 0);
  
      if (scale < 1.0f)
      {
        ctx_reset_path (ctx);
        ctx_rectangle (ctx, 0, 0, width, height);
        ctx_listen (ctx, CTX_DRAG, raise_to_second, client, NULL);
        ctx_reset_path (ctx);
      }
  
      ctx_restore (ctx);
    }
    else
    {
      ctx_client_use_images (ctx, client);
    }

  }

  if (client_count > 1)
  {

#if 0
    ctx_rectangle (ctx, ctx_virtual_width (ctx)/2 + font_size * 0.5,
    0 , font_size * 3, font_size * 3);
    ctx_listen (ctx, CTX_CLICK, raise_to_top, second, NULL);
    ctx_reset_path (ctx);
    ctx_rgba (ctx, .7,.7,.7,1);
    icon (ctx, ctx_virtual_width(ctx)/2 + font_size * 0.5,
               //ctx_virtual_height(ctx) * 0.34,
               0,//ctx_virtual_height(ctx) * 0.34,
        2 * font_size, 2 * font_size, "switch", 

"lineWidth 0.2 lineCap round lineJoin round rgba 0 0 0 0.25 M 0.2 0.45 m 0.15 0.15 l -0.15 -0.15 l 0.15 -0.15 l -0.15 0.15 l 0.6 0.0 l -0.15 0.15 l 0.15 -0.15 l -0.15 -0.15 l 0.15 0.15 stroke rgba 1 1 1 0.5 lineWidth 0.08 M 0.2 0.45 m 0.15 0.15 l -0.15 -0.15 l 0.15 -0.15 l -0.15 0.15 l 0.6 0.0 l -0.15 0.15 l 0.15 -0.15 l -0.15 -0.15 l 0.15 0.15 stroke ");
#endif

  }

  ctx_restore (ctx);
}


void block_events (CtxEvent *event, void *a, void *b)
{
  ctx_event_stop_propagate (event);
}

uint8_t *font_list = NULL;
uint8_t *font_list_mono = NULL;

typedef struct CtxSpan {const char *rstr;
int start;
int end; // points at one beyond end;
} CtxSpan;

CtxSpan ctx_span_new (const char *ro_string)
{
  int length = strlen (ro_string);
  return (CtxSpan){ro_string, 0, length};
}

int ctx_span_length (CtxSpan span)
{
  return span.end - span.start;
}

CtxSpan ctx_span_subspan (CtxSpan span, int start, int end)
{
  CtxSpan ret = span;
  ret.start+=start;
  ret.end = (end-start) + ret.start;
  if (ret.end > span.end)
    ret.end = span.end;
  return ret;
}

int ctx_span_has_prefix (CtxSpan span, const char *prefix)
{
  int prefix_len = strlen (prefix);
  if (ctx_span_length (span) < prefix_len) return 0;
  return !strncmp (&span.rstr[span.start], prefix, prefix_len);
}

int ctx_span_cmp_str (CtxSpan span, const char *str)
{
  if (!str)
    return 0;
  int len = strlen (str);
  if (ctx_span_length (span) != len) return 0;
  return !strncmp (&span.rstr[span.start], str, len);
}

int ctx_span_cmp_span (CtxSpan span, CtxSpan spanb)
{
  int len = ctx_span_length (spanb);
  if (ctx_span_length (span) != len) return 0;
  return !strncmp (&span.rstr[span.start], &spanb.rstr[spanb.start], len);
}

int ctx_span_has_suffix (CtxSpan span, const char *suffix)
{
  int suffix_len = strlen (suffix);
  if (ctx_span_length (span) < suffix_len) return 0;
  return !strncmp (&span.rstr[span.end-suffix_len-1], suffix, suffix_len);
}

CtxSpan ctx_span_chop (CtxSpan *span, char delimiter)
{
  CtxSpan ret = *span;
  for (int i = span->start; i< span->end; i++)
  {
    if (span->rstr[i] == delimiter)
    {
      span->start = i+1;
      ret.end = i;
      return ret;
    }
  }
  span->start = span->end = 0;
  return ret;
}

CtxSpan ctx_span_trim (CtxSpan span, const char *set)
{
  int match = 0;
  do {
  match = 0;
  for (int i = 0; set[i]; i++)
  {
    if (span.rstr[span.start]==set[i])
      match = 1;
  }

  if (match && span.start<span.end)
  {
    span.start++;
  } else match = 0;

  } while (match);


  do {

  match = 0;
  for (int i = 0; set[i]; i++)
  {
    if (span.rstr[span.end-1]==set[i])
      match = 1;
  }

  if (match && span.end>span.start)
  {
    span.end--;
  } else match = 0;

  } while (match);

  return span;
}

void ctx_span_get_str (CtxSpan span, char *out, int out_buf_len)
{
  int i = 0;
  out[out_buf_len-1] = 0;
  for (i = span.start; i< span.end && i - span.start < out_buf_len-1; i++)
    out[i - span.start] = span.rstr[i];
  out[i - span.start] = 0;
}

void font_path_get_spec (Ctx *ctx, const char *inpath, char **family_ret, char **spec_ret)
{
  CtxSpan rest = ctx_span_new ((char*)font_list);

  while (ctx_span_length (rest) > 0)
  {
    CtxSpan line = ctx_span_chop (&rest, '\n');
    CtxSpan rest = line;
    CtxSpan path = ctx_span_chop (&rest, ',');
    CtxSpan family = ctx_span_chop (&rest, ',');
    CtxSpan sub = ctx_span_chop (&rest, ',');

    //char tmp[256];

    if (ctx_span_cmp_str (path, inpath))
    {
      if (family_ret){
        int len = ctx_span_length (family);
        *family_ret = ctx_malloc (len + 1);
        if (*family_ret)
          ctx_span_get_str (family, *family_ret, len + 1);
      }
      if (spec_ret){
        int len = ctx_span_length (family);
        *spec_ret = ctx_malloc (len + 1);
        if (*spec_ret)
          ctx_span_get_str (sub, *spec_ret, len + 1);
      }
      return;
    }
  }
}

static void set_height (Ctx *ctx, void *userdata, const char *name, int count, float *x, float *y, float *w, float *h)
{
  float *fptr = userdata;
  *h = fptr[0];
}


void toggle_settings (CtxEvent *event, void *a, void *b)
{
  in_settings = 0;
  panel_open = 0;
  ctx_queue_draw (event->ctx);
}


void set_hue (int temp, float val)
{
  char name[16];
  float rgb[3];
  ui_hue_to_rgb (val, rgb);
  sprintf (name, "#%02x%02x%02x", (int)(rgb[0]*255),(int)(rgb[1]*255),(int)(rgb[2]*255));
  if (temp < 0)
    cfg_set_string (&ctx_config, "style", "cool", name, NULL);
  else
    cfg_set_string (&ctx_config, "style", "warm", name, NULL);
  cfg_reset_cache (&ctx_config);
}

enum {
  STYLE_CUSTOM = 0,
  STYLE_DARK = 1,
  STYLE_LIGHT = 2,
  STYLE_LIGHT_HC = 3,
  STYLE_DARK_HC = 4,
  STYLE_EINK = 5,
};

void update_ui_colors (Ui *ui)
{
  float cool_hue   = cfg_f32 (&ctx_config, "style", "cool-hue");
  float warm_hue   = cfg_f32 (&ctx_config, "style", "warm-hue");
  float brightness = cfg_f32 (&ctx_config, "style", "brightness");
  float contrast   = cfg_f32 (&ctx_config, "style", "contrast");
  float saturation = cfg_f32 (&ctx_config, "style", "saturation");
  float base_saturation = cfg_f32 (&ctx_config, "style", "base-saturation");

  set_hue (-1, cool_hue);
  set_hue (1, warm_hue);

  uint8_t rgb[4];
  float   rgbf[4];
  cfg_rgba (&ctx_config, "style", "warm", rgb);
  for (int i = 0; i < 3; i ++)
    rgbf[i]=rgb[i]/255.0f;
  ui_color_set_warm (ui, rgbf);
  cfg_rgba (&ctx_config, "style", "cool", rgb);
  for (int i = 0; i < 3; i ++)
    rgbf[i]=rgb[i]/255.0f;
  ui_color_set_cool (ui, rgbf);
  cfg_rgba (&ctx_config, "style", "brightest", rgb);
  for (int i = 0; i < 3; i ++)
    rgbf[i]=rgb[i]/255.0f;
  ui_color_set_brightest (ui, rgbf);
  cfg_rgba (&ctx_config, "style", "darkest", rgb);
  for (int i = 0; i < 3; i ++)
    rgbf[i]=rgb[i]/255.0f;
  ui_color_set_darkest (ui, rgbf);

  ui_set_brightness (ui, brightness);
  ui_set_contrast (ui, contrast);
  ui_set_saturation (ui, saturation);
  ui_set_base_saturation (ui, base_saturation);
  update_vts_light_mode (ui_ctx(ui));

  tabs_update_theme (ui);
}


      int style_variant (Ui *ui)
      {
      int preset = 0;

      if (
        cfg_f32 (&ctx_config, "style", "brightness") == STYLE_EINK_BRIGHTNESS &&
        cfg_f32 (&ctx_config, "style", "contrast") == STYLE_EINK_CONTRAST &&
        cfg_f32 (&ctx_config, "style", "saturation") == STYLE_EINK_SATURATION)
          preset = STYLE_EINK;
      if (
        cfg_f32 (&ctx_config, "style", "brightness") == STYLE_DARK_BRIGHTNESS &&
        cfg_f32 (&ctx_config, "style", "contrast") == STYLE_DARK_CONTRAST &&
        cfg_f32 (&ctx_config, "style", "saturation") == STYLE_DARK_SATURATION)
          preset = STYLE_DARK;
      if (
        cfg_f32 (&ctx_config, "style", "brightness") == STYLE_LIGHT_BRIGHTNESS &&
        cfg_f32 (&ctx_config, "style", "contrast") == STYLE_LIGHT_CONTRAST &&
        cfg_f32 (&ctx_config, "style", "saturation") == STYLE_LIGHT_SATURATION)
          preset = STYLE_LIGHT;
      if (
     cfg_f32 (&ctx_config, "style", "brightness") == STYLE_DARK_HC_BRIGHTNESS &&
     cfg_f32 (&ctx_config, "style", "contrast") == STYLE_DARK_HC_CONTRAST &&
     cfg_f32 (&ctx_config, "style", "saturation") == STYLE_DARK_HC_SATURATION)
          preset = STYLE_DARK_HC;
      if (
     cfg_f32 (&ctx_config, "style", "brightness") == STYLE_LIGHT_HC_BRIGHTNESS &&
     cfg_f32 (&ctx_config, "style", "contrast") == STYLE_LIGHT_HC_CONTRAST &&
     cfg_f32 (&ctx_config, "style", "saturation") == STYLE_LIGHT_HC_SATURATION)
          preset = STYLE_LIGHT_HC;

      ui_choice_add (ui, "custom", 0);
      ui_choice_add (ui, "dark high contrast", STYLE_DARK_HC);
      ui_choice_add (ui, "dark", STYLE_DARK);
      ui_choice_add (ui, "light", STYLE_LIGHT);
      ui_choice_add (ui, "high contrast", STYLE_LIGHT_HC);
      ui_choice_add (ui, "eink", STYLE_EINK);

      int old_preset = preset;
      preset = ui_choice (ui, "style variant", preset);


      if (old_preset != preset)
      switch (preset)
      {
        case STYLE_DARK:
          cfg_set_f32 (&ctx_config, "style", "brightness", STYLE_DARK_BRIGHTNESS, NULL);
          cfg_set_f32 (&ctx_config, "style", "contrast",   STYLE_DARK_CONTRAST, NULL);
          cfg_set_f32 (&ctx_config, "style", "saturation", STYLE_DARK_SATURATION, NULL);
          update_ui_colors (ui);
          break;
     case STYLE_LIGHT:
       cfg_set_f32 (&ctx_config, "style", "brightness", STYLE_LIGHT_BRIGHTNESS, NULL);
       cfg_set_f32 (&ctx_config, "style", "contrast",   STYLE_LIGHT_CONTRAST, NULL);
       cfg_set_f32 (&ctx_config, "style", "saturation", STYLE_LIGHT_SATURATION, NULL);
       update_ui_colors (ui);
       break;
     case STYLE_LIGHT_HC:
        cfg_set_f32 (&ctx_config, "style", "brightness", STYLE_LIGHT_HC_BRIGHTNESS, NULL);
        cfg_set_f32 (&ctx_config, "style", "contrast",   STYLE_LIGHT_HC_CONTRAST, NULL);
        cfg_set_f32 (&ctx_config, "style", "saturation", STYLE_LIGHT_HC_SATURATION, NULL);
        update_ui_colors (ui);
        break;
     case STYLE_DARK_HC:
        cfg_set_f32 (&ctx_config, "style", "brightness", STYLE_DARK_HC_BRIGHTNESS, NULL);
        cfg_set_f32 (&ctx_config, "style", "contrast",   STYLE_DARK_HC_CONTRAST, NULL);
        cfg_set_f32 (&ctx_config, "style", "saturation", STYLE_DARK_HC_SATURATION, NULL);
        update_ui_colors (ui);
        break;
     case STYLE_EINK:
       cfg_set_f32 (&ctx_config, "style", "brightness", STYLE_EINK_BRIGHTNESS, NULL);
       cfg_set_f32 (&ctx_config, "style", "contrast",   STYLE_EINK_CONTRAST, NULL);
       cfg_set_f32 (&ctx_config, "style", "saturation", STYLE_EINK_SATURATION, NULL);
       update_ui_colors (ui);
      break;
    default:
      break;
  }
  return preset;
}

void terminal_settings (Ctx *ctx)
{
  float fs = font_size;

  static float settings_w = 1024.0f;

  float settings_w_desired = (ctx_virtual_width (ctx) - panel_width (ctx) / 4) * 0.95;



  if (settings_w_desired > fs * 50)
    settings_w_desired = fs * 50;

  if (fabsf (settings_w_desired - settings_w) > 1)
  {
  settings_w = settings_w * 0.95 + 0.05 * settings_w_desired;
  ctx_queue_draw (ctx);
  }

  float settings_x =


 ((ctx_virtual_width (ctx) - panel_width (ctx) / 4) - settings_w)/2;



  float settings_y = (ctx_virtual_height (ctx) * 0.1);

  ctx_save (ctx); // a

  if (!font_list)
  {
    if (system("fc-list | sort > /tmp/fc_list")){}
    ctx_get_contents ("/tmp/fc_list", &font_list, NULL);
    for (int i = 0; font_list[i]; i++)
      if (font_list[i]==':') font_list[i]=','; // hack we want just one separator
  }
  if (!font_list_mono)
  {
    if (system("fc-list :mono | sort > /tmp/fc_list_mono")){}
    ctx_get_contents ("/tmp/fc_list_mono", &font_list_mono, NULL);
    for (int i = 0; font_list_mono[i]; i++)
      if (font_list_mono[i]==':') font_list_mono[i]=','; // hack we want just one separator
  }

  //ctx_translate (ctx, settings_x, settings_y);
  ctx_deferred_round_rectangle (ctx, "bg", settings_x, settings_y, settings_w, ctx_virtual_height (ctx)/2-fs*2, fs);

  ui_color_base (ui, 0.5, cfg_f32 (&ctx_config, "style", "background-temperature"));

  ctx_listen (ctx, CTX_DRAG, block_events, NULL, NULL);
  ctx_preserve (ctx);
  ctx_fill (ctx);

  ui_color_ink (ui, 0.5, 0);

  ctx_stroke (ctx);
  ui_color_ink (ui, 0.8, 0.0);
  ctx_save (ctx); //b

  ui_set_dim (ui, settings_x + fs, settings_y + fs, em * 7.5, fs * 40);

  ui_choice_add (ui, "ui", TAB_UI);
  ui_choice_add (ui, "input", TAB_INPUT);
  ui_choice_add (ui, "rendering", TAB_RENDERING);
#if CTX_FONT_ENGINE_HARFBUZZ
  if (ctx_backend_type (ctx) != CTX_BACKEND_CTX)
    ui_choice_add (ui, "fonts", TAB_FONTS);
#endif
  ui_choice_add (ui, "colors", TAB_COLORS);
  ui_choice_add (ui, "about", TAB_ABOUT);
  ui_choice_add (ui, "license", TAB_LICENSE);
  ui_choice_add (ui, "help", TAB_HELP);

  ui_flags (ui, UI_NO_LABEL|UI_NO_BASE|UI_INLINE);
  active_tab = ui_choice (ui, "category", active_tab);


  ui_set_dim (ui, settings_x + em * 9, settings_y + fs, settings_w - em * 9.5, fs * 40);


  switch (active_tab)
  {
  case TAB_FONTS:
    {
  char *fonts[]={ // "Regular",
                  "Mono",
                  "Mono Bold",
                  "Mono Italic",
                  "Mono Bold Italic", 
                  NULL, };
  for (int font_slot = 0; fonts[font_slot]; font_slot++)
  {
    int family = -1;
    int fam_t = -1;
    char *target_family = NULL;
    char *font_path = cfg_string_full (&ctx_config, "fonts", fonts[font_slot], "", NULL);
    int no = 0;
    //CtxSpan rest = ctx_span_new (font_slot>0?(char*)font_list_mono:(char*)font_list);
    CtxSpan rest = ctx_span_new ((char*)font_list_mono);
    CtxSpan prev_family = ctx_span_new("");


    no = 0;
    while (ctx_span_length (rest) > 0)
    {
      CtxSpan line = ctx_span_chop (&rest, '\n');
      CtxSpan rest = line;
      CtxSpan path = ctx_span_chop (&rest, ',');
      CtxSpan fam = ctx_span_chop (&rest, ',');
      //CtxSpan sub = ctx_span_chop (&rest, ',');

      char tmp[256];

      ctx_span_get_str (path, tmp, 255);
      if (font_path[0] && tmp[0] && !strcmp (font_path, tmp) && family == -1)
      {
        family = fam_t = no;
        ctx_span_get_str (fam, tmp, 255);
        if (target_family)
          ctx_free (target_family);
        target_family = ctx_strdup (tmp);
      }
      no ++;
    }


    char label[64];
    sprintf (label, "%s family", fonts[font_slot]);

    ui_choice_add (ui, "<builtin>", -1);


    //rest = ctx_span_new (font_slot>0?(char*)font_list_mono:(char*)font_list);
    rest = ctx_span_new ((char*)font_list_mono);
    no = 0;
    while (ctx_span_length (rest) > 0)
    {
      CtxSpan line = ctx_span_chop (&rest, '\n');
      CtxSpan rest = line;
      CtxSpan path = ctx_span_chop (&rest, ',');
      CtxSpan fam = ctx_span_chop (&rest, ',');
      //CtxSpan sub = ctx_span_chop (&rest, ',');

      char tmp[256];

      ctx_span_get_str (path, tmp, 255);
      if (!ctx_span_cmp_span (fam, prev_family))
      {
        ctx_span_get_str (fam, tmp, 255);
        int len = strlen(tmp);
        if (len<250)
        {
          tmp[len+2]=0;
          tmp[len+1]=no;
          tmp[len]='\b';
        }

        if (target_family && ctx_span_cmp_str (fam, target_family))
          ui_choice_add (ui, tmp, fam_t);
        else
          ui_choice_add (ui, tmp, no);
      }
      prev_family = fam;
      no ++;
    }
    int new_family = ui_choice (ui, label, family);

    int variant = new_family;
    int new_variant = variant;

    if (new_family != -1)
    {
      char label[64];
      sprintf (label, "%s variant", fonts[font_slot]);


    //CtxSpan rest = ctx_span_new (font_slot>0?(char*)font_list_mono:(char*)font_list);
    CtxSpan rest = ctx_span_new ((char*)font_list_mono);

    no = 0;
    while (ctx_span_length (rest) > 0)
    {
      CtxSpan line = ctx_span_chop (&rest, '\n');
      CtxSpan rest = line;
      CtxSpan path = ctx_span_chop (&rest, ',');
      //CtxSpan family = ctx_span_chop (&rest, ',');
      char tmp[256];

      ctx_span_get_str (path, tmp, 255);
      if (variant == -1 && font_path[0] && !strcmp (font_path, tmp))
        variant = no;
      no++;
    }


      //rest = ctx_span_new (font_slot>0?(char*)font_list_mono:(char*)font_list);
      rest = ctx_span_new ((char*)font_list_mono);
      no = 0;

    while (ctx_span_length (rest) > 0)
    {
      CtxSpan line = ctx_span_chop (&rest, '\n');
      CtxSpan rest = line;
      CtxSpan path = ctx_span_chop (&rest, ',');
      CtxSpan family = ctx_span_chop (&rest, ',');
      CtxSpan sub = ctx_span_chop (&rest, ',');

      char tmp[256];

      ctx_span_get_str (path, tmp, 255);

      if (ctx_span_cmp_str (family, target_family))
      {
        ctx_span_get_str (sub, tmp, 255);
        int len = strlen(tmp);
        if (len<250)
        {
          tmp[len+2]=0;
          tmp[len+1]=no;
          tmp[len]='\b';
        }
        ui_choice_add (ui, tmp, no);
      }
      no++;
    }
    new_variant = ui_choice (ui, label, variant);

      

    }

    if (new_variant != variant || (new_family != family))
    {
      //CtxSpan rest = ctx_span_new (font_slot>0?(char*)font_list_mono:(char*)font_list);
      CtxSpan rest = ctx_span_new ((char*)font_list_mono);
  
      int no = 0;
      int didset = 0;
      while (ctx_span_length (rest) > 0)
      {
        CtxSpan line = ctx_span_chop (&rest, '\n');
        CtxSpan rest = line;
        CtxSpan path = ctx_span_chop (&rest, ',');
  
        if (no == new_variant) 
        {
          char tmp[256];
          ctx_span_get_str (path, tmp, 255);
          //fprintf (stderr, "setting %s to %s (%i)\n", fonts[font_slot], tmp, new_variant);
          cfg_set_string (&ctx_config, "fonts", fonts[font_slot], tmp, NULL);
          didset = 1;
        }
        no ++;
      }
      if (!didset)
      {
          //fprintf (stderr, "setting %s to builtin\n", fonts[font_slot]);
          cfg_set_string (&ctx_config, "fonts", fonts[font_slot], "", NULL);
      }
      load_fonts(ctx);
    }

    if (target_family)
      ctx_free (target_family);
  }


  }

    break;
  case TAB_INPUT:
  {
    ui_title (ui, "on screen keyboard");

  int on_screen_keyboard_new = ui_toggle (ui, "on screen keyboard", on_screen_keyboard);

  if (on_screen_keyboard_new != on_screen_keyboard)
  {
    toggle_keyboard (ctx);
  }

  float nosk_width = ui_slider (ui, "keyboard size", 20.0f, 100.0f, 2.0, osk_width);
  if (nosk_width != osk_width)
  {
    update_maximized_size (ctx);
    osk_width = nosk_width;
  }

    ui_title (ui, "security");
  int internal_clipboard = cfg_boolean (&ctx_config, "ctx", "internal_clipboard");
  int new_state = ui_toggle (ui, "disable system clipboard, doesn't allow pasting to external webbrowser, but also means internally copied text between tabs are not leaking to other apps with access to system clipboard contents", internal_clipboard);

  if (new_state != internal_clipboard)
  {
    cfg_set_boolean (&ctx_config, "ctx", "internal_clipboard", new_state, NULL);
    ctx_internal_clipboard (ctx, new_state);
  }


  }
  break;
  case TAB_RENDERING:
  {
    ui_text (ui, "these settings are for testing only and are not persisted in the configuration.");

  int flags = ctx_cb_get_flags (ctx);
  int active = 0;
#define flag_toggle(label, flag) \
  active = (flags & flag) != 0; \
  active = ui_toggle (ui, label, active);\
  if (active) \
    flags |= (flag); \
  else \
    flags &= ~(flag);
  //flag_toggle("mark damage", CTX_FLAG_DAMAGE_CONTROL);
  flag_toggle("show fps", CTX_FLAG_SHOW_FPS);
  flag_toggle("RGB subpixel rendering, done as a fill-in for changed tiles when two equal frames are seen, all content is rendered at 3 times horizontal and vertical resolution and scaled down appropriately for the subpixels.", CTX_FLAG_SUBPIXEL);
  flag_toggle("render animations at quarter resolution", CTX_FLAG_LOWFI);

  flag_toggle("software cursor", CTX_FLAG_POINTER);

  ctx_cb_set_flags (ctx, flags);
  }break;
  case TAB_HELP:
    {

ui_text (ui, "Keybindings:\n"
"\n"
" alt-1..9 - select tab\n"
" alt-0 - swap front tabs\n"
"\n"
" shift-control-C copy selection\n"
" shift-control-V paste\n"
"\n"
" shift-control-up scroll up\n"
" shift-control-down scroll down\n"
" shift-control-right open panel\n"
"\n"
" shift-control-T new tab\n"
" shift-control-W close tab\n"
" shift-control-M (un)maximize\n"
" shift-control-F (un)fullscreen (host)\n"
"\n"
" shift-control-P preferences\n"
" shift-control-H help\n"
//" shift-control-D toggle dark mode\n"
" shift-control-Q quit\n"
"\n"
" control+= increase text size\n"
" control-- decrease text size\n"

"\n"
" shift-control-L toggle magnifying loupe\n"
" shift-control-K toggle on screen keyboard\n"
" shift-control-E floating window mode\n");
     ui->y = em * 30;
    }
    break;

  case TAB_COLORS:
    {

#define STYLE_COLOR(name) \
    { ui_flags (ui, UI_NO_ALPHA);\
     char *new_str= ui_color_popup (ui, name, \
         cfg_string (&ctx_config, "style", name));\
     if (new_str) \
     { \
       cfg_set_string (&ctx_config, "style", name, new_str, NULL); \
       cfg_reset_cache (&ctx_config); \
       ctx_free (new_str);\
       update_ui_colors (ui);\
     }\
    }
#define STYLE_FLOAT(name, min, max, step) {\
    float val = cfg_f32 (&ctx_config, "style", name);\
    float new_val = ui_slider (ui, name, min, max, step, val);\
    if (val != new_val){\
      cfg_set_f32 (&ctx_config, "style", name, new_val, NULL);\
      update_ui_colors (ui);\
    }}\

    STYLE_COLOR("wallpaper");
    int preset = style_variant (ui);

    if (preset == 0)
    {
      STYLE_FLOAT("brightness",0,100,5);
      STYLE_FLOAT("saturation",0,300,5);
      STYLE_FLOAT("contrast",1.0,8,0.5);
    }

    if (preset != 5)
    {

      ui_flags (ui, UI_INLINE);
    if (ui_button (ui, "reset colors"))
    {
      cfg_set_f32 (&ctx_config, "style", "background-temperature", 0.0f, NULL);
      cfg_set_f32 (&ctx_config, "style", "base-saturation", 50.0f, NULL);
      cfg_set_f32 (&ctx_config, "style", "warm-hue", 0.1f, NULL);
      cfg_set_f32 (&ctx_config, "style", "cool-hue", 0.56f, NULL);
      cfg_set_string (&ctx_config, "style", "brightest", "#f", NULL);
      cfg_set_string (&ctx_config, "style", "darkest", "#0", NULL);
      update_ui_colors (ui);
    }
      ui_flags (ui, UI_INLINE);
    if (ui_button (ui, "swap hues"))
    {
      float cool = cfg_f32 (&ctx_config, "style", "warm-hue");
      float warm = cfg_f32 (&ctx_config, "style", "cool-hue");
      cfg_set_f32 (&ctx_config, "style", "warm-hue", warm, NULL);
      cfg_set_f32 (&ctx_config, "style", "cool-hue", cool, NULL);
      update_ui_colors (ui);
    }
    ui_newline (ui);


    STYLE_FLOAT("cool-hue",0.0,1.0,0.05f);
    STYLE_FLOAT("warm-hue",0.0,1.0,0.05f);

    //STYLE_FLOAT("background-temperature",-0.5,0.5,0.05);
    //STYLE_FLOAT("base-saturation",0,100.0,5.0);


    //STYLE_COLOR("warm");
    //STYLE_COLOR("cool");
    STYLE_COLOR("brightest");
    STYLE_COLOR("darkest");


      int rows = 7;
      int cols = 7;
      float cw = em;
      float ch = em;
      float gap = -em * 0.05;

      for (int row = 0; row < rows; row ++)
      for (int col = 0; col < cols; col ++)
      {
        ctx_rectangle (ui->ctx, ui->x0 + col * cw,
                                ui->y + row * ch, cw-gap, ch-gap);
        float value = (col/(cols-1.0f) + row/(rows-1.0f))/2.0f;
        float temp = (col/(cols-1.0f) + (rows-1-row)/(rows-1.0f))-1.0f;

        ui_color (ui, UI_COLOR_FULL, value, temp, 1.0f);
        ctx_fill (ui->ctx);
      }

      for (int row = 0; row < rows; row ++)
      for (int col = 0; col < cols; col ++)
      {
        ctx_rectangle (ui->ctx, ui->x0 + (1*(cols+1)+col) * cw,
                                ui->y + row * ch, cw-gap, ch-gap);
        float value = (col/(cols-1.0f));
        float temp = (((rows-1-row)/(rows-1.0f))-0.5f)*2;

        ui_color_base (ui, value, temp);
        ctx_fill (ui->ctx);
      }

      for (int row = 0; row < rows; row ++)
      for (int col = 0; col < cols; col ++)
      {
        ctx_rectangle (ui->ctx, ui->x0 + (2*(cols+1)+col) * cw,
                                ui->y + row * ch, cw-gap, ch-gap);
        float value = (col/(cols-1.0f));
        float temp = (((rows-1-row)/(rows-1.0f))-0.5f)*2;

        ui_color_ink (ui, value, temp);
        ctx_fill (ui->ctx);
      }

      ui->y += em * rows;
    }
    }
    break;

  case TAB_UI:
  {

  //static char *test = NULL;
  //ui_entry_realloc (ui, "foo", "fallback", &test);
  //
  //



    {
    int val = ctx_get_fullscreen (ctx);
    int nval = ui_toggle (ui, "fullscreen", val);
    if (nval != val)
      toggle_fullscreen (ctx);
    }


#if CTX_FONT_ENGINE_HARFBUZZ
    ctx_vt_enable_ligatures = ui_toggle (ui, "coding ligatures", ctx_vt_enable_ligatures);
#endif

    float new_font_size = ui_slider (ui, "text size", 4.0, 64.0, 0.25f, font_size);


    int preset =
      style_variant (ui);
    if (preset == 0)
      STYLE_FLOAT("brightness",0,100,5);


    animation_duration = ui_slider (ui, "animation duration", 0.0f, 1.0f, 0.05f, animation_duration);

  {
 
  float new_line_spacing = ui_slider (ui, "terminal line spacing", 0.8, 2.0, 0.025, line_spacing);
  float new_baseline = baseline;//ui_slider (ui, "baseline", -1.0, 2.0, 0.025, baseline);


  if (new_font_size != font_size || new_line_spacing != line_spacing || new_baseline != baseline)
  {
    cfg_set_f32 (&ctx_config, "ctx", "line_spacing", new_line_spacing, NULL);
    line_spacing = new_line_spacing;
    cfg_set_f32 (&ctx_config, "ctx", "baseline", new_baseline, NULL);
    baseline = new_baseline;
    set_new_font_size (ctx, new_font_size);
  }


  }



  }break;

  case TAB_ABOUT:
    {
    const char *backend_name = ((CtxBackend*)ctx_get_backend (ctx))->name;
    if (!backend_name) backend_name = "unset";

    ui_textf (ui, "ctx %i.%i.%i\nvector graphics terminal",
    ctx_get_major_version(),ctx_get_minor_version(),ctx_get_micro_version());
    ui_textf (ui, "backend: %s", backend_name);
    ui_textf (ui, "size: %iX%i (%ix%i)", (int)ctx_virtual_width (ctx), (int)ctx_virtual_height (ctx), (int)ctx_width (ctx), (int) ctx_height (ctx));

    int no = 0;
    const char *ev_src = NULL;
    do {
    ev_src = ctx_event_source_name (ctx, no);

    if (ev_src)
    {
      ui_textf (ui, "evsrc: %s", ev_src);
      no++;
    }
    } while (ev_src);

    ui_textf (ui, "configuration is stored in\n%s", ctx_term_config_path ());

    ui_text (ui, "Copyright (c) 2012-2025 Øyvind Kolås <pippin@gimp.org> with contributors; under the ISC license, for details see license tab\n");

    }
    break;
  case TAB_LICENSE:
    {
      static int module = 0;

      ui_choice_add (ui, "ctx", 0);
#if CTX_HARFBUZZ
      ui_choice_add (ui, "harfbuzz", 2);
#endif
#ifdef CTX_STB_IMAGE
      ui_choice_add (ui, "stb_image", 1);
#endif
      ui_choice_add (ui, "miniz", 3);
#if CTX_CURL
      ui_choice_add (ui, "curl", 4);
#endif
      module = ui_choice (ui, "module", module);

      switch (module)
      {

#ifdef CTX_STB_IMAGE
        case 1:
    ui_text (ui, "stb_image - v2.30 - public domain image loader");
    break;
#endif

        case 3:
    ui_text (ui, "miniz - MIT licensed - Copyright 2013-2014 RAD Game Tools and Valve Software Copyright 2010-2014 Rich Geldreich and Tenacious Software LLC\n");
    break;

        case 0:
    {
    ui_title (ui, "license");

    const char *backend_name = ((CtxBackend*)ctx_get_backend (ctx))->name;
    if (!backend_name) backend_name = "unset";

    ui_textf (ui, "ctx %i.%i.%i\nvector graphics terminal", ctx_get_major_version(),ctx_get_minor_version(),ctx_get_micro_version());
    ui_text (ui, "Copyright (c) 2012, 2015, 2019, 2020, 2021, 2022, 2023, 2024, 2025 Øyvind Kolås <pippin@gimp.org> with contributors.\n");
    ui_text (ui, "Permission to use, copy, modify, and/or distribute this software for any purpose with or without fee is hereby granted,\nprovided that the above copyright notice and this permission notice appear in all copies.\n"
 
"THE SOFTWARE IS PROVIDED \"AS IS\" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.\nIN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,\nARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.");

    }
        break;
        case 2:
#if CTX_HARFBUZZ
    ui_text (ui, "harfbuzz - MIT licensed - Copyright 2012-2025 	Behdad Esfahbod, Khaled Hosny et al\n");
#endif
    break;

#if CTX_CURL
        case 4:
    ui_text (ui, "contains curl");
    ui_text (ui, 

"Copyright (c) 1996 - 2025, Daniel Stenberg, daniel@haxx.se, and many contributors, see the THANKS file.\n"

"All rights reserved.\n"

"Permission to use, copy, modify, and distribute this software for any purpose with or without fee is hereby granted, provided that the above copyright notice and this permission notice appear in all copies.\n"

"THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.\n"

"Except as contained in this notice, the name of a copyright holder shall not be used in advertising or otherwise to promote the sale, use or other dealings in this Software without prior written authorization of the copyright holder.\n");

    break;
#endif

      }
    break;
  }
  }

  float result_h = ui->y + em * 1;
  if (result_h < em * 16) result_h = em * 16;
  ctx_resolve (ctx, "bg", set_height, &result_h);

  ctx_restore (ctx); // b
  ctx_restore (ctx); // a

  //ctx_add_key_binding (ctx, "escape", "hide settings", "", toggle_settings, NULL);
}

void ctx_unix_start (Ctx *ctx);
void ctx_unix_stop  (Ctx *ctx);
int ctx_unix_status  (Ctx *ctx);
void ctx_tcp_start (Ctx *ctx);
void ctx_tcp_stop  (Ctx *ctx);
int ctx_tcp_status  (Ctx *ctx);

void monitor_motion (CtxEvent *event, void *a, void *b)
{
  ctx_queue_draw (event->ctx);
}

#if CTX_BIN_BUNDLE
int ctx_terminal_main (int argc, char **argv)
#else
int main (int argc, char **argv)
#endif
{
  execute_self = argv[0];
  const char *commandline = NULL;
  float global_scale = 1.0;

#if 0
  int is_charging = 0;
  float battery_level = battery_status (&is_charging);
  printf ("battery level: %f %s\n", battery_level, is_charging?"charging":"");
#endif

  {
    ctx_config_timestamp = path_modtime (ctx_term_config_path());
    cfg_load (&ctx_config, ctx_term_config_path (),
"# ctx-configuration, this file is kept in sync with ctx at runtime\n"
);
    {
      char *ver = cfg_string (&ctx_config, "ctx", "version");
      if (ver)
      {
        if (!strcmp (ver, CTX_VERSION_STRING))
        {
           // mostly remove old things?
           // we for that no specific check is needed just remove any old cruft?
        }
        free (ver);
      }
    }

    cfg_add_category (&ctx_config, "ctx", "ctx settings");
    cfg_set_string (&ctx_config, "ctx", "version", CTX_VERSION_STRING, NULL);
    cfg_add_category (&ctx_config, "fonts",
"Loaded \"system\" fonts, fallbacks exist for Regular and Mono when left as empty\n# strings. Valid values are full paths to OTF, TTF or ctx font files.");
    cfg_add_category (&ctx_config, "style", "Visual/UI appearance.");
    cfg_add_category (&ctx_config, "palette", "terminal color palette");
    //cfg_add_category (&ctx_config, "system", "misc global settings");
  }
        
  osk_width = cfg_f32_full (&ctx_config, "osk", "width", osk_width, "on screen keyboard size");

  int height = cfg_int_full (&ctx_config, "ctx", "height", 1080, NULL);
  int width  = cfg_int_full (&ctx_config, "ctx", "width",  1920, "window width in pixels, -1 for backend default");
  int fullscreen = cfg_boolean_full (&ctx_config, "ctx", "fullscreen", 1, NULL);
  int monitor_config = cfg_boolean_full (&ctx_config, "", "monitor_config", 1, "at a performance cost, check for timestamp changes on config file");
  int cols = -1;


  if (getpid () == 1)
  {
    int ign;
    ign = system ("pkill plymouth"); // might be hogging stdin
    ign = system ("mount -o remount,rw /");
    ign = system ("mount -a");
    if (ign) {};
  }

  for (int i = 1; argv[i]; i++)
  {
    if (!strcmp (argv[i], "--help"))
    {
      printf ("ctx-terminal [--width px] [--height px] [--cols target columns] [--font-size size] [-e [--] command]\n");
      return 0;
    }
    else if (!strcmp (argv[i], "-e"))
    {
      commandline = argv[i+1];
      i++;
      if (!strcmp (commandline, "--"))
      {
        i++;
        commandline_argv_start = i;
      }
      terminal_no_new_tab = 1;
    }
    else if (!strcmp (argv[i], "--fullscreen"))
      fullscreen = 1;
    else if (!strcmp (argv[i], "--no-save-config"))
      no_save_config = 1;
    else if (!strcmp (argv[i], "--width"))
        width = consume_float (argv, &i);
    else if (!strcmp (argv[i], "--height"))
        height = consume_float (argv, &i);
    else if (!strcmp (argv[i], "--cols"))
        cols = consume_float (argv, &i);
    else if (!strcmp (argv[i], "--font-size"))
        font_size = consume_float (argv, &i) * global_scale;
    else if (argv[i][0]=='-')
    {
      switch (argv[i][1])
      {
        case 's': font_size = consume_float (argv, &i) * global_scale; break;
        case 'c': cols = consume_float (argv, &i); break;
        case 'w': width = consume_float (argv, &i); break;
        case 'h': height = consume_float (argv, &i); break;
      }
    }
  }

  if (font_size > 0 && width <0 && cols < 0)
  {
    cols = 80;
    width = font_size * cols / 2;
    height = font_size * 25;
  }

  height = cfg_int_full (&ctx_config, "ctx", "height", height, NULL);
  width  = cfg_int_full (&ctx_config, "ctx", "width",  width, "window width in pixels, -1 for backend default");

  ctx  = ctx_new (width, height, NULL);
  load_config(ctx);
  if (fullscreen)
  {
    ctx_set_fullscreen (ctx, 1);
    layout_no = LAYOUT_FOUR_BY_THREE;
  }
  else
  {
    layout_no = LAYOUT_MAXIMIZED;
  }
  width = ctx_width (ctx);
  height = ctx_height (ctx);

  if (layout_no == LAYOUT_WINDOWS)
    ctx_set_focus_cb (ctx, wm_focus_cb, NULL);
  else
    ctx_set_focus_cb (ctx, NULL, NULL);
 
  vt_set_ctx_events_init (terminal_ctx_events_init, ui);

#if CTX_SOCKETS
  ctx_unix_start (ctx);
  ctx_tcp_start (ctx);
#endif


#if CTX_TERM && CTX_TERMINAL_EVENTS
  if (ctx_backend_type (ctx) == CTX_BACKEND_TERM && font_size <= 0)
  {
    font_size = ctx_term_get_cell_height (ctx);
  }
#endif

  if (cols <= 0)
  {
    //if (((double)(width))/height < (16.0/9.0))
    //  cols = 80;
    //else
      cols = 80;
  }

  if (font_size < 0)
    font_size = floorf (width / cols  * 2 / 1.5);

  //void *itk = NULL;

  ctx_font_size (ctx, font_size);

  ui = ui_for_ctx (ctx);
  if (commandline_argv_start)
  {
    add_tab_argv (ctx, &argv[commandline_argv_start], 1);
  }
  else
  {
    if (!commandline)
      commandline = ctx_find_shell_command();
    add_tab (ui, ctx, commandline, 1);
  }

#if GNU_C
  int mt = ctx_add_timeout (ctx, 1000 * 20, malloc_trim_cb, NULL);
#endif


  float prev_ms = ctx_ms (ctx);
  int cfg_has_reloaded = 1;

  update_ui_colors (ui);
  while (ctx_clients (ctx) && !ctx_has_exited (ctx))
    {
      //int n_clients = ctx_list_length (ctx_clients (ctx));
      float ms = ctx_ms (ctx);
      float delta_s = (ms - prev_ms)/1000.0f;
      prev_ms = ms;

      full_time += delta_s;
      if (cfg_has_reloaded)
      {
        load_config (ctx);
        cfg_has_reloaded = 0;
      }

      {
        static int lock_control = 0;
        lock_control--;

        if (lock_control < 0)
        {
          int lastval = locked;
          if (access(ctx_term_lock_path (), R_OK) == F_OK)
          {
            locked = 1;
          }
          else
          {
            locked = 0;
          }
          if (locked != lastval)
          {
            ctx_queue_draw (ctx);
            if (locked)
            {
              char *command = NULL;
              if (access ("/.flatpak-info", F_OK) != -1)
              {
                command = ctx_strdup_printf ("flatpak-spawn --env=TERM=xterm --host "
                "sh -c \"echo 'ctx is locked';"
                "while [ -f %s ];do su `whoami` -c 'rm %s';done\"",
                 ctx_term_lock_path(), ctx_term_lock_path());
              }
              else
              {
                command = ctx_strdup_printf ("sh -c \"echo 'ctx is locked';"
              "while [ -f %s ];do su `whoami` -c 'rm %s';done\"",
              ctx_term_lock_path(), ctx_term_lock_path());
              }

              add_tab (ui, ctx, command, 0);
              ctx_free (command);
              if (!on_screen_keyboard)
                on_screen_keyboard = 2;
            }
            else
            {
              if (on_screen_keyboard == 2)
                on_screen_keyboard = 0;
            }
          }
          lock_control = 50;
        }

      }

      if (ctx_need_redraw(ctx))
      {
        float elapsed = ctx_start_frame (ctx);
        ctx_font_size (ctx, font_size);
        ui_start_frame (ui);

        if (!no_save_config && ctx_cfg_dirty != 0.0f)
        {
          ctx_cfg_dirty += elapsed;
          if (ctx_cfg_dirty > 4.0)
          {
            sync_config ();
            cfg_save (&ctx_config, ctx_term_config_path ());
            ctx_config_timestamp = path_modtime (ctx_term_config_path());
          }
        }

#if 1
        if (monitor_config &&
            (ctx_config_timestamp != path_modtime (ctx_term_config_path())))
        {
          ctx_wait_for_renderer (ctx);
          usleep (200 * 1000);
          ctx_config_timestamp = path_modtime (ctx_term_config_path());
          cfg_load (&ctx_config, ctx_term_config_path (), NULL);
          monitor_config = cfg_boolean (&ctx_config, "", "monitor_config");
          font_size = cfg_f32 (&ctx_config, "ctx", "font_size");
          line_spacing = cfg_f32 (&ctx_config, "ctx", "line_spacing");
          baseline = cfg_f32 (&ctx_config, "ctx", "baseline");
          animation_duration = cfg_f32 (&ctx_config, "style", "animation_duration");

          #if 1
          for (CtxList *l = ctx_clients (ctx); l; l = l->next)
          {
            CtxClient *client = l->data;
            VT *vt = ctx_client_vt (client);
            vt_set_font_size (vt, font_size);
            vt_set_baseline (vt, baseline);
            vt_set_line_spacing (vt, line_spacing);
            vt_recompute_cols (vt);
          }
          ctx_queue_draw (ctx);
          #endif
          cfg_has_reloaded = 1;
        }
  #endif


        //ctx_rgb (ctx, 0.086,0.086,0.113);
        ctx_font_size (ctx, font_size);
        for (int round = 0; round < (magnifier_active?2:1); round++)
        {

        ctx_save (ctx);

          if (round == 1)
          {
            float factor = 2.4;
            float radius = ctx_virtual_height (ctx) / 2.5;
            ctx_rectangle (ctx, 0,0, ctx_width(ctx), ctx_height(ctx));
            ctx_listen (ctx, CTX_MOTION, monitor_motion, NULL, NULL);
            //ctx_rectangle (ctx, ctx_pointer_x(ctx) - 200, ctx_pointer_y (ctx) - 200, 400, 400);
            ctx_reset_path (ctx);
            ctx_arc (ctx, ctx_pointer_x(ctx), ctx_pointer_y(ctx), radius + 2, 0, M_PI*2, 0);
            ui_color_ink (ui, 1, 0);
            ctx_line_width (ctx, 3);
            ctx_stroke (ctx);
            ctx_arc (ctx, ctx_pointer_x(ctx), ctx_pointer_y(ctx), radius, 0, M_PI*2, 0);
            ctx_clip (ctx);
            ctx_rgb (ctx,0,0,0);
            ctx_paint (ctx);

            ctx_translate (ctx,  ctx_pointer_x(ctx), ctx_pointer_y(ctx));
            ctx_scale (ctx, factor, factor);
            ctx_translate (ctx,  -ctx_pointer_x(ctx), -ctx_pointer_y(ctx));
 
          }

#if ENABLE_PORTRAIT
        //if (is_portrait())
        {
          ctx_translate (ctx, ctx_width(ctx)/2, ctx_height(ctx)/2);
          float rot_width  = ctx_virtual_width (ctx);
          float rot_height = ctx_virtual_height (ctx);
          ctx_translate (ctx, 
               (rot_height - ctx_height(ctx)) /2.0f,
               -(rot_width - ctx_width(ctx)) /2.0f
            );
          ctx_rotate (ctx, ctx_virtual_rotation (ctx));
          ctx_translate (ctx, -ctx_width(ctx)/2, -ctx_height(ctx)/2);
        }
#endif
        //ctx_rectangle (ctx, 0, 0, ctx_width (ctx), ctx_height (ctx));
        //ctx_fill (ctx);
        //ctx_paint (ctx);

        if (locked)
        {
          ctx_term_lock_screen (ctx);
        }
        else
        {
          switch (layout_no)
          {
            case LAYOUT_MAXIMIZED:
              maximized (ctx, elapsed);
              break;
            case LAYOUT_WINDOWS:
              window_manager (ctx, elapsed);
              break;
            default:
            case LAYOUT_FOUR_BY_THREE:
              four_by_three (ctx, elapsed);
              break;
          }
        }

        if (in_settings)
        {
          terminal_settings (ctx);
        }

        //if (ctx_get_fullscreen (ctx))
        //
        // maybe kms/fb should always report as fullscreen?
        ctx_osk_draw (ctx);

        if (!locked)
        {
          draw_panel (ctx);
          osk_button (ctx);
        }

        ui_draw_overlays (ui);
        ctx_restore (ctx);
        
        }

        if (backlight != 100.0f)
        {
          ctx_rgba (ctx, 0.0,0.0,0.0, (100.0f-backlight)/100.0f);
          ctx_paint (ctx);
        }

        if (!in_settings)
        {
          ctx_listen (ctx, CTX_KEY_PRESS, terminal_key_any, ui, NULL);
          ctx_listen (ctx, CTX_KEY_DOWN,  terminal_key_any, ui, NULL);
          ctx_listen (ctx, CTX_KEY_UP,    terminal_key_any, ui, NULL);
          ctx_listen (ctx, CTX_TEXT_INPUT,terminal_key_any, ui, NULL);
        }

#if 0 
        ctx_reset_path (ctx);
        ctx_rectangle (ctx, 0, 0, ctx_virtual_width (ctx), ctx_virtual_height (ctx));


        if (cfg_boolean (&ctx_config, "", "auto_hide_mouse"))
          ctx_listen (ctx, CTX_MOTION, terminal_show_cursor, NULL, NULL);
#endif
        ctx_reset_path (ctx);

        ui_end_frame (ui);
  
        ctx_add_key_binding (ctx, "shift-control-i", "screenshot", "", screenshot_action_cb, NULL);
        ctx_add_key_binding (ctx, "shift-control-p", "toggle settings", "", toggle_settings, NULL);
        ctx_add_key_binding (ctx, "shift-control-k", "toggle keyboard", "", toggle_keyboard_cb, NULL);
        ctx_add_key_binding (ctx, "shift-control-l", "toggle magnifier", "", toggle_magnifier_cb, NULL);
        ctx_add_key_binding (ctx, "shift-control-d", "toggle light mode", "", toggle_light_mode_cb, NULL);

        ctx_end_frame (ctx);
      }
      else
      {
        ctx_handle_events (ctx);
      }

     {
       int active_id = ctx_clients_active (ctx);
       CtxClient *active = active_id>=0?ctx_client_by_id (ctx, active_id):NULL;
       if (active)
         terminal_update_title (ctx_client_title (active));
     }
    }

  ui_destroy (ui);
#if CTX_SOCKETS
  ctx_unix_stop (ctx);
  ctx_tcp_stop (ctx);
#endif

#if GNU_C
  ctx_remove_idle (ctx, mt);
#endif

  if (!no_save_config)
  {
    sync_config ();
    cfg_save (&ctx_config, ctx_term_config_path ());
  }

  ctx_destroy (ctx);

  if (getpid () == 1)
  {
    if (system ("reboot --force")){};
  }

  return 0;
}

#else

#if CTX_BIN_BUNDLE
int ctx_terminal_main (int argc, char **argv)
#else
int main (int argc, char **argv)
#endif
{
  return -1;
}

#endif
