#include "ctx.h"
#include <time.h>
#include <sys/time.h>
#include <math.h>
#include <unistd.h>

static void draw_analog_clock_at_time (Ctx *ctx,
                                       float x, float y, float radius,
                                       const char *config,
                                       uint64_t ms_since_midnight)
{
  int second_smooth = strstr(config, "ss") != NULL;
  int second_highlight = strstr(config, "sh") != NULL;
  int second_line = strstr(config, "sl") != NULL;
  int second_dot  =  strstr(config, "sd") != NULL;
  int mark_minute = strstr(config, "mm") != NULL;
  int mark_noon = strstr(config, "mn") != NULL;
  int mark_hour = strstr(config, "mh") != NULL;
  int mark_cardinal =strstr(config, "mc") != NULL;
  uint64_t ms = ms_since_midnight;
  uint32_t s = ms / 1000;
  uint32_t m = s / 60;
  uint32_t h = m / 60;

  ctx_save(ctx);

  ms = ((uint64_t)(ms)) % 1000;
  s %= 60;
  m %= 60;
  h %= 12;

  if (second_smooth == 0)
      ms = 0;

  float r;

  ctx_line_width(ctx, radius * 0.02f);
  ctx_line_cap(ctx, CTX_CAP_ROUND);

  ctx_save(ctx);
  ctx_global_alpha(ctx, 0.5f);
  for (unsigned int markm = 0; markm < 60; markm++) {
    r = markm * CTX_PI * 2 / 60.0 - CTX_PI / 2;

    if (markm == s && second_highlight) {
      ctx_global_alpha(ctx, 1.0f);
    } else
      ctx_global_alpha(ctx, 0.33f);

    if (((markm % 5) == 0)) {
      if (mark_minute || mark_hour || (mark_cardinal && (markm==0 || markm==15 || markm==30 || markm==45)) || (mark_noon && markm==0)
          )
      {
        ctx_move_to(ctx, x + cosf (r) * radius * 0.8f,
                  y + sinf (r) * radius * 0.8f);
        ctx_line_to(ctx, x + cosf(r) * radius * 0.95f,
                  y + sinf (r) * radius * 0.95f);
        ctx_stroke(ctx);
      }
    } else {
      if (mark_minute)
      {
        ctx_move_to(ctx, x + cosf (r) * radius * 0.92f,
                    y + sinf (r) * radius * 0.92f);
        ctx_line_to(ctx, x + cosf (r) * radius * 0.95f,
                    y + sinf (r) * radius * 0.95f);
        ctx_stroke(ctx);
      }
    }
  }
  ctx_restore (ctx);
  ctx_global_alpha (ctx, 0.75f);

  ctx_line_width (ctx, radius * 0.075f);
  ctx_line_cap (ctx, CTX_CAP_ROUND);

  r = m * CTX_PI * 2 / 60.0 - CTX_PI / 2;

  ctx_move_to(ctx, x, y);
  ctx_line_to(ctx, x + cosf(r) * radius * 0.7f, y + sinf(r) * radius * 0.7f);
  ctx_stroke(ctx);

  r = (h + m / 60.0) * CTX_PI * 2 / 12.0 - CTX_PI / 2;
  ctx_move_to(ctx, x, y);
  ctx_line_to(ctx, x + cosf(r) * radius * 0.4f, y + sinf(r) * radius * 0.4f);
  ctx_stroke(ctx);

  ctx_line_width(ctx, radius * 0.02f);
  ctx_line_cap(ctx, CTX_CAP_NONE);

  r = (s + ms / 1000.0f) * CTX_PI * 2 / 60 - CTX_PI / 2;

  ctx_global_alpha(ctx, 0.5f);
  if (second_line)
  {
    ctx_move_to(ctx, x, y);
    ctx_line_to(ctx, x + cosf(r) * radius * 0.78f, y + sinf(r) * radius * 0.78f);
    ctx_stroke(ctx);
  }
  if (second_dot)
  {
    ctx_arc(ctx, x + cosf(r) * radius * 0.85f, y + sinf(r) * radius * 0.85f, 
                 radius * 0.025, 0.0, M_PI*2, 0);
    ctx_fill(ctx);
  }

  ctx_restore(ctx);
}

void draw_analog_clock (Ctx *ctx, float x, float y, float radius, const char *config)
{
  time_t now = time (NULL);
  struct tm *bt = localtime (&now);
  struct timeval t;
  gettimeofday(&t, NULL);
  uint64_t ms_since_midnight =
        ((bt->tm_hour * 60 + bt->tm_min) * 60 + bt->tm_sec) * 1000 + 
        + t.tv_usec / 1000;

  draw_analog_clock_at_time (ctx, x, y, radius, config, ms_since_midnight);
}

void exit_cb (CtxEvent *event, void *a, void *b)
{
  ctx_exit (event->ctx);
}

int idle_queue_draw (Ctx *ctx, void *data)
{
  ctx_queue_draw (ctx);
  return 0;
}

void clock_loop (Ctx *ctx, float delta_s, void *data)
{
  const char *config = data;
  int second_smooth = strstr (config, "ss") != NULL;
  if (second_smooth)
  {
    ctx_queue_draw (ctx);
  }
  else
  {
    ctx_add_timeout (ctx, 500, idle_queue_draw, NULL);
  }

  float w = ctx_width(ctx);
  float h = ctx_height(ctx);
  float min = w < h ? w : h;

  ctx_rgba (ctx, 0,0,0,1.0f);
  ctx_paint (ctx);
  ctx_rgba (ctx, 1,1,1,1);

  draw_analog_clock (ctx, w / 2, h / 2, min / 2, config);

  ctx_add_key_binding (ctx, "q", "exit", "leave view", exit_cb, NULL);
  ctx_add_key_binding (ctx, "escape", "exit", "leave view", exit_cb, NULL);
  ctx_add_key_binding (ctx, "backspace", "exit", "leave view", exit_cb, NULL);
}

int main (int argc, char **argv){
  char *config = "mm;mh;ss;sd ";
  if (argv[1]) config = argv[1];

  ctx_main (NULL, clock_loop, config);

  return 0;
}
