typedef enum {
 CONFIG_NONE,
 CONFIG_INT,
 CONFIG_FLOAT,
 CONFIG_BOOLEAN,
 CONFIG_STRING,
 CONFIG_QSTRING,
} ConfigType;

ConfigType cfg_type    (char **config, const char *category, const char *key);

float  cfg_set_f32     (char **config, const char *category, const char *key, float value, const char *comment);
double cfg_set_f64     (char **config, const char *category, const char *key, double value, const char *comment);
int    cfg_set_int     (char **config, const char *category, const char *key, int value, const char *comment);
const char *cfg_set_string  (char **config, const char *category, const char *key, const char *value, const char *comment);
int cfg_set_boolean (char **config, const char *category, const char *key, int value, const char *comment);

char  *cfg_string  (char **config, const char *category, const char *key);
int    cfg_boolean (char **config, const char *category, const char *key);
int    cfg_int     (char **config, const char *category, const char *key);
float  cfg_f32     (char **config, const char *category, const char *key);
double cfg_f64     (char **config, const char *category, const char *key);

int   cfg_add_category   (char **config, const char *category, const char *comment);
int   cfg_category_count (char **config);
int   cfg_key_count      (char **config, const char *category);
char *cfg_category_name  (char **config, int no);
char *cfg_key_name       (char **config, const char *category, int no);

int   cfg_has_key       (char **config, const char *category, const char *key);

void  cfg_ctx_color_full (char **config, const char *category, const char *key, Ctx *ctx, const char *default_value, const char *comment);
void  cfg_ctx_color      (char **config, const char *category, const char *key, Ctx *ctx);


int   cfg_rgba_full (char **config, const char *category, const char *key, uint8_t *rgba, const char *default_value, const char *comment);
int   cfg_rgba (char **config, const char *category, const char *key, uint8_t *rgba);


char  *cfg_string_full  (char **config, const char *category, const char *key, const char *default_value, const char *comment);
int    cfg_boolean_full (char **config, const char *category, const char *key, int default_value, const char *comment);
int    cfg_int_full     (char **config, const char *category, const char *key, int default_value, const char *comment);
float  cfg_f32_full     (char **config, const char *category, const char *key, float default_value, const char *comment);
double cfg_f64_full     (char **config, const char *category, const char *key, double default_value, const char *comment);

#define CFG (*config)   // used as if is a string pointer in the code; all functions take
                        // the pointer to a string pointer as first argument

#include <stdlib.h>
int cfg_load (char **config, const char *cfg_path, const char *default_value)
{
  if (*config)
  {
    free (*config);
    *config = NULL;
  }
  if (!*config)
  {
    uint8_t *contents = NULL;
    ctx_get_contents (cfg_path, &contents, NULL);
    if (contents)
    {
      *config = (char*)contents;
      return 0;
    }
    else
    {
      if (default_value)
        *config = strdup (default_value);
      else
        *config = strdup ("");
    }
  }
  return -1;
}

static void make_ancestors (const char *path)
{
  char *apath = strdup (path);
  for (int i = 1; apath[i]; i++)
  {
    if (apath[i] == '/')
    {
      apath[i] = 0;
      mkdir (apath, 0777);
      apath[i] = '/';
    }
  }
  free (apath);
}

static void cfg_save (char **config, const char *cfg_path)
{
  if (!*config)
    return;

  if (ctx_cfg_dirty == 0.0f)
    return;
  ctx_cfg_dirty = 0.0f;

  make_ancestors (cfg_path);
  int cfg_path_len = strlen (cfg_path);
  char *tmp_path = malloc (cfg_path_len+3);
  if (!tmp_path)
    return;
  strcpy(tmp_path, cfg_path);
  tmp_path[cfg_path_len]='~';
  tmp_path[cfg_path_len+1]=0;

  FILE *f = fopen (tmp_path, "w");
  if (f)
  {
    fwrite (*config, 1, strlen(CFG), f);
    fclose (f);
    rename(tmp_path, cfg_path);
  }

  free (tmp_path);
}

static int cfg_resolve_category (char **config, const char *category)
{
  if (category == NULL) return 0;
  if (!*config) return -1;
  int pos = 0;

  if (category[0] == 0)
  {
    return pos;
  }
  else
  {
    int catlen = strlen (category);
    while (CFG[pos] && !(CFG[pos]=='[' && CFG[pos+catlen+1]==']' && !strncmp(category, &CFG[pos+1], catlen)))
    {
      if (CFG[pos]=='#' || 
          CFG[pos]=='=')
        while (CFG[pos] && CFG[pos]!='\n')
          pos ++;
      pos++;
    }

    if (strncmp(category, &CFG[pos+1], catlen))
      return -1;
    while (CFG[pos] && CFG[pos]!='\n')pos++;
    return pos + 1;
  }
}

static int cfg_resolve (char **config, const char *cat, const char *key)
{
  if (cat == NULL) cat = "";
  if (!*config)
    return -1;

  int pos = 0;

  pos = cfg_resolve_category (config, cat);
  if (pos < 0)
    return -2;

  if (!key)
    return pos;

  int keylen = strlen (key);
  char key_start = key[0];

  while (CFG[pos])
  {
     while (CFG[pos] && !((pos == 0 ||  ( CFG[pos-1] == ' ' || CFG[pos-1] == '\n')) &&
                          CFG[pos]==key_start &&
                          (!strncmp(key, &CFG[pos], keylen))))
     {
        if (CFG[pos]=='[')
          return -1;
        if (CFG[pos]=='#' || 
            CFG[pos]=='=')
           while (CFG[pos] && CFG[pos]!='\n')
             pos ++;
        pos++;
     }
     if (CFG[pos]==key_start &&
         !strncmp(key, &CFG[pos], keylen))
     {
       pos += keylen;
       while (CFG[pos] == ' ') pos++;
       if (CFG[pos]=='=')
       {
         pos++;
         while (CFG[pos]==' ') pos++;
         return pos;
       }
       else
       {
         pos ++;
       }
     }
     else
     {
        pos ++;
     }
  }
  return -1;
}

ConfigType cfg_type_for_str (char *str)
{
  int is_num = 0;
  int pos = 0;

  switch (str[pos])
  {
    case '0': case '1': case '2': case '3': case '4': case '5':
    case '6': case '7': case '8': case '9': case '-': case '.':
      is_num = 1;
      break;
    case 't':
    case 'f':
      return CONFIG_BOOLEAN;
    case '"':
      return CONFIG_STRING;
    case '\'':
      return CONFIG_QSTRING;
    default:
      return CONFIG_NONE;
  }

  if (!is_num)
    return CONFIG_NONE;
 
  int decimals = 0;
  while ( 
       (str[pos] >='0' && str[pos] <='9') |
       (str[pos] =='.') ||
       (str[pos] =='-')){
       if (str[pos] == '.') decimals = 1;
       pos ++;
   }

  if (decimals) return CONFIG_FLOAT;
  return CONFIG_INT;
}

static ConfigType cfg_type_at_pos (char **config, int pos)
{
  if (pos < 0)
    return CONFIG_NONE;
  return cfg_type_for_str (&CFG[pos]);
}

static int cfg_value_len_at_pos (char **config, int pos)
{
  ConfigType type = cfg_type_at_pos (config, pos);
  int ret = 0;
  if (type == CONFIG_NONE)
    return 0;
  if (type == CONFIG_STRING)
  {
    ret++;pos++;
    while (CFG[pos] != '"')
    {
      if (CFG[pos] && CFG[pos]=='\\' && CFG[pos+1])
      {
        ret++;
        pos++;
      }
      ret++;
      pos++;
    }
    ret++;
    return ret;
  }
  else if (type == CONFIG_QSTRING)
  {
    ret++;pos++;
    while (CFG[pos] && CFG[pos] != '\'')
    {
      ret++;
      pos++;
    }
    ret++;
    return ret;
  }

  while (CFG[pos] != ' ' && CFG[pos] != '\n' && CFG[pos]!='#')
  {
    ret ++;
    pos ++;
  }
  return ret;
}

ConfigType cfg_type (char **config, const char *cat, const char *key)
{
  return cfg_type_at_pos (config, cfg_resolve (config, cat, key));
}

int cfg_add_category (char **config, const char *cat, const char *comment)
{
  int pos = cfg_resolve_category (config, cat);
  if (pos < 0)
  {  // create category
     int conflen = strlen (CFG);
     char *temp = malloc (conflen + strlen (cat) + 15 + (comment?strlen(comment):0));
     strcpy (temp, CFG);
     pos = conflen;

     temp[pos++] = '\n';
     if (comment)
     {
       temp[pos++] = '#';
       temp[pos++] = ' ';
       for (int i = 0; comment[i];i++)
         temp[pos++] = comment[i];
       temp[pos++] = '\n';
     }

     temp[pos++] = '[';
     strcpy (&temp[pos], cat);
     pos+=strlen(cat);
     temp[pos++] = ']';
     temp[pos++] = '\n';
     temp[pos] = 0;
     void *t=CFG;
     CFG = temp;
     free (t);
     conflen = strlen (CFG);
  }
  return pos;
}

static void cfg_set_rawstr (char **config, const char *category, const char *key, const char *value, const char *comment)
{
  int pos = cfg_resolve (config, category, key);
  if (pos < 0)
   {
     pos = cfg_add_category (config, category, NULL);

     int conflen = strlen (CFG);

     if (comment)
     {
        int orig_pos = pos;
        char *temp = malloc (conflen + strlen (comment) + 15);
        strcpy (temp, CFG);
        //if (pos >0 && temp[pos] != '\n')
        //  temp[pos++] = '\n';
        temp[pos++] = '#';
        temp[pos++] = ' ';
        strcpy (&temp[pos], comment);
        pos+=strlen(comment);
        temp[pos++] = '\n';
        strcpy (&temp[pos], &CFG[orig_pos]);
        void *t=CFG;
        CFG = temp;
        free (t);
        conflen = strlen (CFG);
     }

     // create item
     {
        char *temp = malloc (conflen + strlen (key) + 5);
        int keylen = strlen (key);
        memcpy (temp, CFG, pos);
        memcpy (&temp[pos], key, keylen);
        pos+=keylen;

        temp[pos]   = ' ';
        temp[pos+1] = '=';
        temp[pos+2] = ' ';
        temp[pos+3] = '\n';

        memcpy (&temp[pos+4], &CFG[pos-keylen], conflen-(pos-keylen));
        temp[pos+4 + conflen-(pos-keylen)] = 0;
        pos += 3;
        void *t=CFG;
        CFG = temp;
        free (t);
     }
   }

  int oldlen = cfg_value_len_at_pos (config, pos);
  int newlen = strlen (value);

  if (oldlen == newlen && !memcmp (&CFG[pos], value, oldlen))
  {
    return;
  }
  int conflen = strlen (CFG);

  if (oldlen != newlen)
  {
    char *temp = malloc (conflen - oldlen + newlen + 2);
    memcpy (temp, CFG, pos);
    memcpy (&temp[pos + newlen], &CFG[pos + oldlen], conflen - (pos + oldlen));
    memcpy (&temp[pos], value, newlen);
    temp[conflen-oldlen+newlen]=0;
    free (CFG);
    CFG = temp;
  }
  else
  {
    memcpy (&CFG[pos], value, newlen);
  }
  ctx_cfg_dirty += 0.1f;
}

double cfg_set_f64 (char **config, const char *cat, const char *key, double value, const char *comment)
{
  char str[32];
  sprintf(str, "%f", value);
  int len = 0;
  len = strlen (str);
  while (len > 3 && str[len-1] == '0' && str[len-2] == '0')
    str[--len]=0;
  cfg_set_rawstr (config, cat, key, str, comment);
  return value;
}

float cfg_set_f32 (char **config, const char *cat, const char *key, float value, const char *comment)
{
  cfg_set_f64 (config, cat, key, value, comment);
  return value;
}

int cfg_set_int (char **config, const char *cat, const char *key, int value, const char *comment)
{
  char str[20];
  sprintf(str, "%d", value);
  cfg_set_rawstr (config, cat, key, str, comment);
  return value;
}

const char *cfg_set_string (char **config, const char *cat, const char *key, const char *value, const char *comment)
{
  int len = 0;
  for (int i = 0; value[i]; i++)
  {
    if (value[i]=='"') len ++;
    if (value[i]=='\\') len ++;
    if (value[i]=='\n') len ++;
    len++;
  }
  char *temp = malloc (len + 1 + 2);
  len = 0;
  temp[len++]='"';
  for (int i = 0; value[i]; i++)
  {
    if (value[i]=='"')
    {
      temp[len++]='\\';
      temp[len++]='"';
    }
    else if (value[i]=='\\')
    {
      temp[len++]='\\';
      temp[len++]='\\';
    }
    else if (value[i]=='\n')
    {
      temp[len++]='\\';
      temp[len++]='n';
    }
    else
    {
      temp[len++]=value[i];
    }
  }
  temp[len++]='"';
  temp[len]=0;
  cfg_set_rawstr (config, cat, key, temp, comment);
  free (temp);
  return value;
}

int cfg_set_boolean (char **config, const char *cat, const char *key, int value, const char *comment)
{
  if (value>0)
    cfg_set_rawstr (config, cat, key, "true", comment);
  else
    cfg_set_rawstr (config, cat, key, "false", comment);
  return value;
}

/* key == NULL means remove category
 */
void cfg_remove (char **config, const char *cat, const char *key)
{
  int pos = cfg_resolve (config, cat, key);
  if (pos < 0)
    return;

  while (pos >0 && CFG[pos]!='=') pos--;
  if (pos >0 && CFG[pos]=='=') pos--;
  while (pos >0 && CFG[pos]==' ') pos--;
  while (pos >0 && CFG[pos]!=' ' && CFG[pos]!='\n') pos--;
  if (pos >0 && (CFG[pos]==' ' || CFG[pos]=='\n')) pos++;

  int linelen = 0;

  int len = strlen (CFG);
  for (int i = 0; pos + i < len && CFG[pos+i] != '\n'; i++) linelen++;
  linelen++;

  if (!key)
  {
    for (int i = linelen; pos + i < len && CFG[pos+i] != '['; i++) linelen++;
  }

  memmove ( &CFG[pos], &CFG[pos + linelen], len - (pos + linelen));
  CFG[pos + (len - (pos + linelen))]=0;

  ctx_cfg_dirty += 0.1f;
}

int cfg_int_full (char **config, const char *cat, const char *key, int default_value, const char *comment)
{
  int pos = cfg_resolve (config, cat, key);
  if (pos < 0) {
    cfg_set_int (config, cat, key, default_value, comment);
    return default_value;
  }

  return atoi(&CFG[pos]);
}

int cfg_int (char **config, const char *cat, const char *key)
{
  return cfg_int_full (config, cat, key, 0, NULL);
}

double cfg_f64_full (char **config, const char *cat, const char *key, double default_value, const char *comment)
{
  int pos = cfg_resolve (config, cat, key);
  if (pos < 0) {
    cfg_set_f64 (config, cat, key, default_value, comment);
    return default_value;
  }

  return atof(&CFG[pos]);
}

double cfg_f64 (char **config, const char *cat, const char *key)
{
  return cfg_f64_full (config, cat, key, 0, NULL);
}

float cfg_f32_full (char **config, const char *cat, const char *key, float default_value, const char *comment)
{
  return cfg_f64_full (config, cat, key, default_value, comment);
}

float cfg_f32 (char **config, const char *cat, const char *key)
{
  return cfg_f64_full (config, cat, key, 0, NULL);
}

char *cfg_string_full (char **config,
                       const char *cat,
                       const char *key,
                       const char *default_value,
                       const char *comment)
{
  int pos = cfg_resolve (config, cat, key);
  if (pos < 0) {
    if (default_value)
    {
      cfg_set_string (config, cat, key, default_value, comment);
      return strdup (default_value);
    }
    return NULL;
  }

  if (CFG[pos]=='\'')
  {
    pos ++;

  int len = 0;
  int tpos = pos;

  while (CFG[tpos] && CFG[tpos]!='\'')
  {
    len++;
    tpos++;
  }
  char *ret = malloc (len + 1);

  len = 0;
  while (CFG[pos] && CFG[pos]!='\'')
  {
    int ch = CFG[pos];
    ret[len++]=ch;
    pos ++;
  }
  ret[len]=0;
  return ret;
  }

  pos ++;
  int len = 0;
  int tpos = pos;

  while (CFG[tpos] && CFG[tpos]!='"')
  {
    len++;
    tpos++;
  }
  char *ret = malloc (len + 1);

  len = 0;
  while (CFG[pos] && CFG[pos]!='"')
  {
    int ch = CFG[pos];

    if (ch == '\\')
    {
       switch(CFG[pos + 1])
       {
         case 'n': 
           ch = '\n';
           break;
         case '\\': 
           ch = '\\';
           break;
         case '"': 
           ch = '"';
           break;
       }
       pos++;
    }
    ret[len++]=ch;
    pos ++;
  }
  ret[len]=0;
  return ret;
}

char *cfg_string (char **config,
                  const char *cat,
                  const char *key)
{
  return cfg_string_full (config, cat, key, NULL, NULL);
}

int cfg_boolean_full (char **config, const char *cat, const char *key, int default_value, const char *comment)
{
  int pos = cfg_resolve (config, cat, key);
  if (pos < 0) {
    if (default_value)
      cfg_set_boolean (config, cat, key, default_value, comment);

    return default_value > 0;
  }
  if (!strncmp (&CFG[pos], "true", 4))
    return 1;
  return 0;
}

int cfg_boolean (char **config, const char *cat, const char *key)
{
  return cfg_boolean_full (config, cat, key, 0, NULL);
}

int   cfg_category_count (char **config)
{
  int pos = 0;
  int count = 0;

  while (CFG[pos])
  {
    if (CFG[pos]=='[')
    {
      pos++;
      count ++;
      while (CFG[pos] && CFG[pos]!=']') pos++;
    }
    if (CFG[pos]=='=' || CFG[pos] == '#')
    {
      while (CFG[pos] && CFG[pos]!='\n') pos++;
    }
    pos++;
  }
  return count;
}

int   cfg_key_count      (char **config, const char *category)
{
  int pos = cfg_resolve_category (config, category);
  if (pos < 0) return 0;

  int count = 0;

  while (CFG[pos])
  {
    if (CFG[pos] == '[')
    {
      return count;
    }
    else if (CFG[pos] == ' ')
    {
    }
    else if (CFG[pos] == '=')
    {
       count++;
       while(CFG[pos] && CFG[pos] !='\n') pos++;
    }
    else if (CFG[pos] == '#')
    {
       while(CFG[pos] && CFG[pos] !='\n') pos++;
    }
    pos++;
  }
  return count;
}

char *cfg_category_name (char **config, int no)
{
  int pos = 0;
  int count = 0;

  while (CFG[pos])
  {
    if (CFG[pos]=='[')
    {
      pos++;
      if (count == no)
      {
        int len = 0;
        while (CFG[pos + len] && CFG[pos + len]!=']') len++;
        char *ret = malloc (len + 1);
        memcpy(ret, &CFG[pos], len);
        ret[len]=0;
        return ret;
      }
      count ++;
      while (CFG[pos] && CFG[pos]!=']') pos++;
    }
    if (CFG[pos]=='=' || CFG[pos] == '#')
    {
      while (CFG[pos] && CFG[pos]!='\n') pos++;
    }
    pos++;
  }
  return NULL;
}

char *cfg_key_name       (char **config, const char *category, int no)
{
  int pos = cfg_resolve_category (config, category);
  if (pos < 0) return NULL;

  char name[64]="";
  int len=0;
  int count = 0;

  int started_line = 0;
  while (CFG[pos])
  
  {
    if (CFG[pos] == '\n')
    {
       len = 0;
       started_line = 0;
    }
    else if (CFG[pos] == '[')
    {
      return NULL;
    }
    else if (CFG[pos] == ' ')
    {
      if (started_line)
      {
       if (len < 62)
         name[len++]=CFG[pos];
      }
    }
    else if (CFG[pos] == '=')
    {
       if (count == no && len)
       {
         name[len]=0;
         while (name[len-1]==' ')
         {
           name[len-1]=0;
           len--;
         }
         return strdup(name);
       }
       count++;
       while(CFG[pos] && CFG[pos] !='\n') pos++;
       len = 0;
       started_line = 0;
    }
    else if (CFG[pos] == '#')
    {
       while(CFG[pos] && CFG[pos] !='\n') pos++;
       len = 0;
       started_line = 0;
    }
    else
    {
       if (len < 62)
         name[len++]=CFG[pos];
       started_line = 1;
    }
    pos++;
  }
  return NULL;
}

int cfg_has_key (char **config, const char *category, const char *key)
{
  return (cfg_resolve (config, category, key) > 0);
}

static inline int ctx_color_hexdigit (char ch)
{  
  if (ch >='0' && ch <= '9')
    return ch - '0';  
  if (ch >='a' && ch <= 'f')
    return ch - 'a' + 10;  
  if (ch >='A' && ch <= 'F')
    return ch - 'a' + 10;
  return 0;
}

typedef struct {
  char *config;
  uint32_t hash;
  uint8_t red;
  uint8_t green;
  uint8_t blue;
  uint8_t alpha;
} CfgColorCache;

typedef struct {
  char *config;
  uint32_t hash;
  float   value;
} CfgValCache;


#define CFG_COLOR_CACHE_SIZE 64

static CfgColorCache color_ht[CFG_COLOR_CACHE_SIZE]={{0,},};

static inline uint32_t cfg_hash (const char *category, const char *key)
{
  return ctx_strhash (category) ^ ctx_strhash (key);
}

void cfg_reset_cache (char **config)
{
  for (int i = 0; i < CFG_COLOR_CACHE_SIZE; i++)
  {
    color_ht[i].hash = 0;
    color_ht[i].config = 0;
  }
}

float cfg_cached_f32_full (char **config, const char *category, const char *key, float default_value, const char *comment)
{
  uint32_t hash = cfg_hash (category, key);
  int pos = hash % CFG_COLOR_CACHE_SIZE;
  if (color_ht[pos].hash == hash && color_ht[pos].config == *config)
    return ((CfgValCache*)color_ht)[pos].value;
  if (color_ht[pos].hash)
  {
    //printf ("clash: %s %s\n", category, key);
  }
  color_ht[pos].hash = hash;
  color_ht[pos].config = *config;
  float value = cfg_f32_full (config, category, key, default_value, comment);
  ((CfgValCache*)color_ht)[pos].value = value;
  return value;
}
float cfg_cached_f32 (char **config, const char *category, const char *key)
{
  return cfg_cached_f32_full (config, category, key, 0.0f, NULL);
}

int   cfg_rgba_full (char **config, const char *category, const char *key, uint8_t *rgba, const char *default_value, const char *comment)
{
  uint32_t hash = cfg_hash (category, key);
  int pos = hash % CFG_COLOR_CACHE_SIZE;
  if (color_ht[pos].hash == hash && color_ht[pos].config == *config)
  {
    do_return:
    if(rgba)
    {
      rgba[0] = color_ht[pos].red;
      rgba[1] = color_ht[pos].green;
      rgba[2] = color_ht[pos].blue;
      rgba[3] = color_ht[pos].alpha;
    }
    return 0;
  }
  if (color_ht[pos].hash)
  {
    //printf ("clash: %s %s\n", category, key);
  }

  char *set = cfg_string_full (config, category, key, default_value, comment);
  if (set)
  {
    if (set)
    {
      int r=0,b=0,g=0,a=255;
      int iscol = 0;
      if (set[0]=='#')
      {
        switch(strlen(set))
        {
          case 2:
            r = ctx_color_hexdigit(set[1]) * 255 / 15;
            g = ctx_color_hexdigit(set[1]) * 255 / 15;
            b = ctx_color_hexdigit(set[1]) * 255 / 15;
            break;
          case 3:
            // hack this up do be a grayscale + alpha one that covers light and
            // dark with a switch?
            {
              int v = ctx_color_hexdigit(set[1]) * 255 / 15;
              r = v;
              g = v;
              b = v;
              a = ctx_color_hexdigit(set[2]) * 255 / 15;
            }
            break;
          case 4:
            r = ctx_color_hexdigit(set[1]) * 255 / 15;
            g = ctx_color_hexdigit(set[2]) * 255 / 15;
            b = ctx_color_hexdigit(set[3]) * 255 / 15;
            break;
          case 5:
            r = ctx_color_hexdigit(set[1]) * 255 / 15;
            g = ctx_color_hexdigit(set[2]) * 255 / 15;
            b = ctx_color_hexdigit(set[3]) * 255 / 15;
            a = ctx_color_hexdigit(set[4]) * 255 / 15;
            break;
          case 7:
            r = ctx_color_hexdigit(set[1]) * 16 + ctx_color_hexdigit(set[2]);
            g = ctx_color_hexdigit(set[3]) * 16 + ctx_color_hexdigit(set[4]);
            b = ctx_color_hexdigit(set[5]) * 16 + ctx_color_hexdigit(set[6]);
            break;
          case 9:
            r = ctx_color_hexdigit(set[1]) * 16 + ctx_color_hexdigit(set[2]);
            g = ctx_color_hexdigit(set[3]) * 16 + ctx_color_hexdigit(set[4]);
            b = ctx_color_hexdigit(set[5]) * 16 + ctx_color_hexdigit(set[6]);
            a = ctx_color_hexdigit(set[7]) * 16 + ctx_color_hexdigit(set[8]);
            break;
        }
        iscol = 1;
      }
      if (iscol)
      {
        color_ht[pos].hash = hash;
        color_ht[pos].config = *config;
        color_ht[pos].red = r;
        color_ht[pos].green = g;
        color_ht[pos].blue = b;
        color_ht[pos].alpha = a;
        free (set);
        goto do_return;
      }
    }
    free (set);
  }
  return -1;
}

int   cfg_rgba (char **config, const char *category, const char *key, uint8_t *rgba)
{
  return cfg_rgba_full (config, category, key, rgba, NULL, NULL);
}

void  cfg_ctx_color_full (char **config, const char *category, const char *key, Ctx *ctx, const char *default_value, const char *comment)
{
  uint8_t rgba[4];
  cfg_rgba_full (config, category, key, rgba, default_value, comment);
  ctx_rgba (ctx, rgba[0]/255.0f, rgba[1]/255.0f, rgba[2]/255.0f, rgba[3]/255.0f);
}

void  cfg_ctx_color (char **config, const char *category, const char *key, Ctx *ctx)
{
  cfg_ctx_color_full (config, category, key, ctx, NULL, NULL);
}
